/*
 * // Copyright (c) Radzivon Bartoshyk 8/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
Taylor expansion at zeros and extremums for J0.

Generated by SageMath:
```python
mp.prec = 180
def print_taylor_coeffs(poly):
    print("[")
    for i in range(0, 24):
        coeff = poly[i]
        print_double_double("", coeff)
    print("],")

def print_taylor_coeffs_dyad(poly):
    print("[")
    for i in range(0, 24):
        coeff = poly[i]
        print_dyadic(coeff)
    print("],")

print(f"pub(crate) static J0_COEFFS_TAYLOR: [[(u64, u64); 24]; {len(j0_zeros)}] = [")

prev_zero = 0

for i in range(0, len(j0_zeros)):
    k_range = j0_zeros[i]
    range_diff = k_range - prev_zero

    mp.prec = 180
    x0 = mp.mpf(k_range)
    from mpmath import mp, j0, taylor
    poly = taylor(lambda val: j0(val), x0, 24)
    print_taylor_coeffs(poly)
    # print(poly)
    prev_zero = j0_zeros[i]

print("];")
```
**/
pub(crate) static J0_COEFFS_TAYLOR: [[(u64, u64); 24]; 47] = [
    [
        (0xb5be985fa7f83cc3, 0xb91cba5b78f9bcd0),
        (0xbc8ac8cc3d6bafa4, 0xbfe09cdb36551280),
        (0xbc5b50b1160e9078, 0x3fbba1deea029494),
        (0x3c4bc98bcfdc7a7f, 0x3facfae864368d70),
        (0xbc1a87a5518aec0c, 0xbf81bb1cbe1a4071),
        (0x3bcf770ef9886bfc, 0xbf61f992590d12bd),
        (0x3ba05114c54b25ab, 0x3f315382ba06cc47),
        (0x3ba2b401cd6c68ec, 0x3f06ed3b9f07eb28),
        (0xbb46fe17c9b10921, 0xbed232c77d228ab6),
        (0x3b479fbd07230860, 0xbea1cce302821846),
        (0x3ac3ee3080fdf4d2, 0x3e67ff99166c20b8),
        (0x3ad99d2c20223b95, 0x3e32951bd4726a93),
        (0xba982691b667ba5c, 0xbdf5c2c38b2a278c),
        (0x3a44e89952bc724c, 0xbdbbdc468c1a817a),
        (0x3a0ebde93f245a29, 0x3d7cd41cf248a22e),
        (0x39aeeb56633d8ce6, 0x3d3f70b0201c12a4),
        (0xb98e8ed54af59db8, 0xbcfd22d8ef70f1aa),
        (0x394a95ebfce4a435, 0xbcbbaa352d8622af),
        (0xb91ff1b0b50e7818, 0x3c773612afd5db30),
        (0xb8dd29c3a1d1250e, 0x3c3382933550f440),
        (0xb8840afcc88fadb8, 0xbbede7dcd5e6dab6),
        (0x384023453882217b, 0xbba68bbe41057f40),
        (0x37e0c3131ee8f030, 0x3b5fce24eb23551c),
        (0x379a6895f9086a05, 0x3b15bc1becd9a431),
    ],
    [
        (0x3c62de1143765a96, 0xbfd9c6cf582cbf7f),
        (0xb56eab9a5fff323e, 0x38d9a91f973ec303),
        (0xbc52de1143765a96, 0x3fc9c6cf582cbf7f),
        (0xbc3b84c5dde43674, 0xbf91f06d14e11e02),
        (0x3c27767d9698b536, 0xbf8b589d1da13905),
        (0x3beae1afe6c3eeda, 0x3f50f9103cf5b152),
        (0x3bd3e8ae8dea10a7, 0x3f386445621cc085),
        (0x3b8a8905b308cb21, 0xbefa2a033ccf2705),
        (0xbb6859e8c0ae37ce, 0xbed83a06e30c4109),
        (0xbb3d54a069e9c64a, 0x3e96a4fd997104b3),
        (0x3af989d32babfca4, 0x3e6ec03c7b7d1357),
        (0xbab1ecf377bfa04f, 0xbe295db03343bc40),
        (0xba90b0b6e5fe7551, 0xbdfb1e242e3fafb5),
        (0x3a3fd469738e9f05, 0x3db3fa9bccb27cd1),
        (0xba1b0f91d98a1070, 0x3d8195cae4f67f9d),
        (0xb9ba66f04112da4a, 0xbd3762afe2a5cc10),
        (0xb99f157bfc6eaf25, 0xbd017a40c9422e17),
        (0x39512dfe72026e3a, 0x3cb52a10c3c6a274),
        (0x390b50bb7a057314, 0x3c7b78e32c0ddf7a),
        (0x38b2f2ab795ef554, 0xbc2e86449ded35db),
        (0x3886128863c137b0, 0xbbf180413173c4a1),
        (0x3830411cc8598621, 0x3ba1f599693c21d5),
        (0x38061e9b17fc3c4e, 0x3b62704bba47e657),
        (0x37ba5ced95be86b1, 0xbb119329ee1d3fb9),
    ],
    [
        (0xb5bc6f55d4dac0c2, 0xb916caed64280f3e),
        (0xbc6af17f78e58353, 0x3fd5c6e60a097823),
        (0xbc22c1940b65933b, 0xbf9f8f72e7a848e0),
        (0x3c39226a6968a5d4, 0xbfab2150cb41e8c1),
        (0xbc0ead94b776cda6, 0x3f72f7ffe90256bb),
        (0xbbdc44bdbe1112fc, 0x3f627e31fe9a9779),
        (0xbbc64599a8123499, 0xbf26f641f41956f7),
        (0xbb948616b6922642, 0xbf0863f481a43036),
        (0x3b680f8935f5af0c, 0x3ecad77d748a06db),
        (0x3b3cc25f06059d1b, 0x3ea32e6d99c6af7d),
        (0xbac3305a73e5ec68, 0xbe62da37e38435b9),
        (0xbad5c613351595f6, 0xbe341d72d9392e0e),
        (0xba9c8106c7bc6fb7, 0x3df1d0433d9a0e49),
        (0x3a54e8ace2031b36, 0x3dbe2f3389aa5f69),
        (0x3a1a0f5d1cba1387, 0xbd78498ffdebdd63),
        (0x39c72f8fafe52f38, 0xbd410477aeae347a),
        (0x398287fc3bba485f, 0x3cf911a8d781ae91),
        (0x395788d424859469, 0x3cbde68af47fa3dd),
        (0xb914472b326b8d7e, 0xbc744b3881dbc1de),
        (0x38d8ef43414cc878, 0xbc350be81f6c25f7),
        (0x388738457ea68eb1, 0x3bea7a3cd386daef),
        (0x381867e5c9a67b59, 0x3ba845cae670510a),
        (0xb7fd1902fcb38c68, 0xbb5c71de4246f315),
        (0xb7bbe92f443184c2, 0xbb175a569f596544),
    ],
    [
        (0x3c7af22d033ee0a4, 0x3fd33518b3874e8a),
        (0xb5a1c201a4cbcd4f, 0x390f8a0b194e92aa),
        (0xbc6af22d033ee0a5, 0xbfc33518b3874e8a),
        (0xbc006437c8a4bc6d, 0x3f7d34125d59d8ff),
        (0x3c276d8715df682d, 0x3f880c83bdeee5b0),
        (0x3be68431910b0ef3, 0xbf4483c20f1cb1cf),
        (0xbbb05a9f39ef055c, 0xbf36ffa5fc8cad74),
        (0x3b913c3053e04d46, 0x3ef2ccf7b21fff81),
        (0x3b34763d026d3263, 0x3ed796a751f89051),
        (0x3b28d27038a5d6ff, 0xbe91e850e4c28d2e),
        (0xbb0dacba8ae06269, 0xbe6e6a49ae425df4),
        (0x3abf3eddd6df5c90, 0x3e254c4387ef6820),
        (0xba918835ce9eaca7, 0x3dfb084d1fe8eb88),
        (0x3a5d71d0cc12f311, 0xbdb177e06f2957d2),
        (0x3a128331bef5dabb, 0xbd8198853f2794c4),
        (0xb9d9afdf19b00f9a, 0x3d350c2b4db26303),
        (0x399d5420c79c50f5, 0x3d0183edbb8ef55e),
        (0x3950904a54f66bdd, 0xbcb376d167cfd12b),
        (0x3900fa2cc4b5e6f2, 0xbc7b8a653f1cd809),
        (0x38c18dc8b4e37444, 0x3c2c8b097fe47227),
        (0x38721faa0c98eabc, 0x3bf189c13e615d1f),
        (0x384e91176ed97cac, 0xbba1041440ddec2d),
        (0x380179051848c349, 0xbb6276b7a23c8925),
        (0x37b217667c662a42, 0x3b10d45d32f264a5),
    ],
    [
        (0xb5a271a164439dcf, 0x39091ad14f5534e3),
        (0x3c70b85158068ef8, 0xbfd15f7977a772d4),
        (0x3c1371b46642ace0, 0x3f900f7fcf183e0d),
        (0xbc44d97f9ccedb81, 0x3fa68b984ec6493c),
        (0x3c042ebf64ab2f7d, 0xbf648e63600d8418),
        (0xbbeee6fe26323549, 0xbf60e0d60385a6f0),
        (0x3baeef0ba4a3d132, 0x3f1d796052775aab),
        (0xbba7aac08a2ef0bb, 0x3f07800bc550673c),
        (0xbb5a88016ef7845c, 0xbec3324842f7d51c),
        (0xbb4a5fc6166e08e4, 0xbea30e8cc35f2086),
        (0x3ae456e8b4cd8ab1, 0x3e5ceda479a13c54),
        (0xbad32f77090cf486, 0x3e34582cb217a0ff),
        (0xba896f49e7d71906, 0xbdecade19f5ddb10),
        (0x3a21bb7dfbc8d1d7, 0xbdbedafbed1c0039),
        (0x3a0e246bb345d838, 0x3d743c230e74f083),
        (0xb9ef2b99a0d63f0b, 0x3d41820d4a50e7ac),
        (0x39930eba3f48ee4f, 0xbcf56d4abfa3f75e),
        (0xb9404cba8724060b, 0xbcbee1c963b55043),
        (0x38e5999953118f29, 0x3c71b059736ac012),
        (0xb8670d7b409cc9b1, 0x3c35c975cfa56031),
        (0x38874e8a77d0b989, 0xbbe76ff93efd4a1a),
        (0xb846172c476991f2, 0xbba9288634dcae38),
        (0xb7c3ad53cba7bf5d, 0x3b597e272f8ae371),
        (0xb793a523f540949e, 0x3b183832a1cf9585),
    ],
    [
        (0xbc5052a3a2541c57, 0xbfcff654544ebcd1),
        (0xb59d2cb904ed26e5, 0xb8f8cdeedd7089d9),
        (0x3c4052a3a2541c57, 0x3fbff654544ebcd1),
        (0xbc1261ff874d8f29, 0xbf70c17ff72afae7),
        (0x3c123a275590a85f, 0xbf84b0c5d5da6789),
        (0xbbda98843bd20bfa, 0x3f394154be70ed46),
        (0x3bd7d4055bf0fe2a, 0x3f34e12c3067bef4),
        (0x3b83b9ba50d44e92, 0xbee9f32fc25ad134),
        (0x3b7dd3415a058d7b, 0xbed63c5475439cb2),
        (0xbb2e30b182db0623, 0x3e8adbafdf1415b1),
        (0xbb03faa98b08aa74, 0x3e6d601d278516f5),
        (0xba8bdda43ca62c5a, 0xbe20eedcab593552),
        (0x3a9e2fad33581bf9, 0xbdfa84fedb589fe9),
        (0xba43dc502b3bc37b, 0x3dacf8e343fe2283),
        (0x3a20042e6b66d5ce, 0x3d81702cfaaee589),
        (0x399e7a30f53ca26c, 0xbd320271d8cd44dc),
        (0x39a5f9316e1c5efa, 0xbd01797e5eacddfb),
        (0xb9547c274818e871, 0x3cb10ed3b5c9be93),
        (0x390484c268c0ce85, 0x3c7b98f45921983a),
        (0x38c24e5982383c83, 0xbc297cbf7079afbf),
        (0x389b277576e91bd0, 0xbbf1a00382f67f4b),
        (0xb81f48467ddb418e, 0x3b9ed8b7eb419122),
        (0x37fcd9d0b0e535bf, 0x3b629700d2c00fd9),
        (0x379a08401f2ad096, 0xbb0ee1e52f6de1ed),
    ],
    [
        (0x35857c582b5ea1c9, 0x390bee0ed8c1329b),
        (0xbc56d72d40e790b3, 0x3fcdc13e66ac2e77),
        (0xbc17a0058564dc78, 0xbf842ff0cdc58463),
        (0xbc4ea7eea5885b06, 0xbfa38d1dd8992e04),
        (0xbbe0000f328100a7, 0x3f5a55e9b346edc1),
        (0xbbf1ad5da35cf264, 0x3f5e2e16f97d0a81),
        (0x3bba0bf2d8b00c54, 0xbf13dfc3782af205),
        (0x3baf5778b0601b70, 0xbf05ce7f496656d0),
        (0x3b554dd7e814f538, 0x3ebbb178da9c4ac0),
        (0x3b4faa40f4ef4735, 0x3ea2346d74940be8),
        (0x3afda3f1a9c16a33, 0xbe5612f2e799b732),
        (0x3ab1ff4f9f08910e, 0xbe33d79882e5df9f),
        (0xba8b82d54967fe42, 0x3de6dbc112bdb279),
        (0x3a54c5a3f521e570, 0x3dbe88c4d898d665),
        (0x3a0aab5610ed8408, 0xbd70ad880fc1e195),
        (0x39ecb4a2ad16ceb5, 0xbd4180d64e2c4131),
        (0xb99c77042f820598, 0x3cf22094137f7e15),
        (0xb94dafad35b07003, 0x3cbf18a8509fa48c),
        (0x390d0551dfd7b655, 0xbc6e8dc12bc24aba),
        (0x38c7f144bd8a3779, 0xbc360ce090718b01),
        (0x3887e3ff32f169c4, 0x3be4947b20d26a23),
        (0x384fdb50bbf01aa8, 0x3ba98e129bc0f8a1),
        (0xb7f91bc574623289, 0xbb56b091d8385139),
        (0x37b564d63f3376cc, 0xbb18aa3f6c83b7e7),
    ],
    [
        (0x3c6c8c66d2e42062, 0x3fcbf3337873a7d8),
        (0xb540e662913e9721, 0xb8b8ae14acc03cc2),
        (0xbc5c8c66d2e42062, 0xbfbbf3337873a7d8),
        (0x3bfaea5f3639483f, 0x3f66604d91f926ee),
        (0xbc24192692d7c9db, 0x3f8251858011816b),
        (0xbbdcdb12750487fd, 0xbf314bc11a32c246),
        (0x3bd50548750ad314, 0xbf32e7decd1f73a5),
        (0xbb847a9ca039b976, 0x3ee293b4c9a24966),
        (0x3b6e3a426e412d42, 0x3ed4a6704d05ad0b),
        (0x3b1253e91cb63be7, 0xbe843ca9b71b6996),
        (0x3afbda58c238f16a, 0xbe6bddfbf35630ba),
        (0x3aba3f365a5e64b5, 0x3e1aae76737061da),
        (0x3a96b049b0e48195, 0x3df992424643ff64),
        (0x3a42ad17879fc5b9, 0xbda7aad4b0989ad6),
        (0xba22956ea0fa52e3, 0xbd81051e84a7a86a),
        (0x39c7f26c2e23f425, 0x3d2e49ae1a0e9424),
        (0xb998466ad52efa61, 0x3d013585a9b7473f),
        (0xb93c2bb2beb0ccde, 0xbcad5c6820d4c19c),
        (0xb90356383e63d305, 0xbc7b5bcfc1f36461),
        (0xb8714991402d4264, 0x3c2659e097d2d299),
        (0x3893b54f82b6acd5, 0x3bf18eefb530cdde),
        (0x380fec88aec6bc4b, 0xbb9b78694703192c),
        (0x37fadce1d8598d3f, 0xbb6296435cd2331f),
        (0xb7a9cd5a8e59cb5f, 0x3b0bdae97028e6a9),
    ],
    [
        (0x35ac59c6428de284, 0x390f184285843311),
        (0x3c32010996eec734, 0xbfca701d0f967500),
        (0xbc1331fb2bff5c30, 0x3f7c54b930fef892),
        (0x3c44240355876f9f, 0x3fa17798aa09f11f),
        (0xbbf91cbfe509a337, 0xbf52a2151407dd09),
        (0xbbeaf2596bf8cd55, 0xbf5b541f829bfb41),
        (0x3b9678567dcb0401, 0x3f0cc0bda19ffe34),
        (0x3b7cc73ae3f8a72f, 0x3f041f3b0662f49e),
        (0x3b50b073a3604389, 0xbeb4b230b114b149),
        (0xbb39f6a1ea249aed, 0xbea1223e2bd0689f),
        (0xbad9dcc2b787e5e9, 0x3e511963968a437f),
        (0xbade1245aa3f79f6, 0x3e32ffb8dedaa0e5),
        (0xba7ddb5234f87c44, 0xbde24c47f1c46f88),
        (0x3a4b24f96f04dc6f, 0xbdbda527540ac89a),
        (0x3a002159e830fd60, 0x3d6b75eeb2587be0),
        (0x39a7a5cb88cfcf0b, 0x3d412dcf9e14b526),
        (0x39536a23b6e364bc, 0xbcee8dc94e61fd9e),
        (0x39406182c6b2e611, 0xbcbec6e9eeb29dec),
        (0xb8fb12be3176058a, 0x3c6a411ab303dfdb),
        (0xb8da27a89b185f00, 0x3c35f7175a19426f),
        (0x388abe97716ef10a, 0xbbe1f8d987025207),
        (0x383e3b32dc10b94d, 0xbba99550856225f2),
        (0xb7e9ddb15c3ea3e0, 0x3b5415925aa1431b),
        (0x3795803255a3ea0c, 0x3b18c98cea89bbf7),
    ],
    [
        (0x3c6e9557ccd1703f, 0xbfc925c6fca08f55),
        (0x35a0840c3eb5c8e9, 0x3909bb79105907c8),
        (0xbc5e9557ccd1703f, 0x3fb925c6fca08f55),
        (0xbbf0bd4a4617bca0, 0xbf6049377403bcb4),
        (0xbc1dac1b118bb946, 0xbf809463bbd0367f),
        (0xbbbd1ccb5b6a209d, 0x3f297b354706c53c),
        (0xbbdfefc73ab9ea4c, 0x3f314dd4404e5fcc),
        (0xbb7689ff8e8b648e, 0xbedbf665d5afcb1e),
        (0xbb55632e0e751036, 0xbed32cb00ee8c1f3),
        (0x3b168e07e41d973b, 0x3e7f50fa815b542a),
        (0xbaeb09f8b87a4826, 0x3e6a4339e5f906ee),
        (0x3ab370c114b8364e, 0xbe1541c07333f998),
        (0x3a95f92ca4cd1f66, 0xbdf86c0bb112e485),
        (0x3a3d7a90841bd326, 0x3da35fba37d1f6f7),
        (0xba2b575752323f67, 0x3d80700770cc15dd),
        (0xb9557a6df0847bf2, 0xbd2963b203086961),
        (0x3994060b8e456ec8, 0xbd00c58770231c0a),
        (0x3934b091af50e5cb, 0x3ca91e0b939f2d33),
        (0xb91ee976639f1136, 0x3c7adb6f69ceb591),
        (0xb8b63616ac9c64e9, 0xbc2373e3cab8e402),
        (0x387455d8787bd080, 0xbbf156093877495b),
        (0x3823f244d844aa0f, 0x3b9842736a626a94),
        (0x37f3628fad20886f, 0x3b626f881139ee38),
        (0x379bb0027a1cd34e, 0xbb08e814b7cab6b6),
    ],
    [
        (0xb59e68fd63ab1a3d, 0xb8fc6246c85a6640),
        (0x3c4a4f96a2520bad, 0x3fc8077f56c9b782),
        (0xbbf97f76587fd9cb, 0xbf75467eb535deaa),
        (0x3c28681fd3a1faac, 0xbf9fd7c3ad6f59e0),
        (0x3be0e0f0ccfb4015, 0x3f4c1b47c809c5eb),
        (0x3bf6295d17a16103, 0x3f59166c7d3eaa7c),
        (0xbba4cf48882a0699, 0xbf05f0152478f5f6),
        (0x3ba861e7d6d89ac5, 0xbf02aa939fd9fc9b),
        (0xbb5e0f27f02ffdbc, 0x3eb0129708ec2410),
        (0x3b4393e44832fa90, 0x3ea01716dc9f2e1b),
        (0x3acfaffff96cb541, 0xbe4b227eef409a9c),
        (0xbadf064b85fdb242, 0xbe32101c49d611c3),
        (0xba4ba7b2c5f1707c, 0x3dddb1e8b7adba5e),
        (0x3a5a73ba24830cae, 0x3dbc80aab1d9322c),
        (0xba00db3fc0b0c1ba, 0xbd66c2b17ab71886),
        (0x39e8d291d0b4d27a, 0xbd40ad5f60b542e8),
        (0x3973527417cabd58, 0x3ce9d02ffd593bab),
        (0xb950911cb567e2a5, 0x3cbe1eda1d51b8b7),
        (0xb8f1f756b53acec7, 0xbc668e76453989dd),
        (0x38c7df6bb93ac8ec, 0xbc35a3dd5c2dd312),
        (0x387c31c12e84cbf7, 0x3bdf55d5df5603a5),
        (0x381a4e01bffe5027, 0x3ba95785174d99bc),
        (0xb7f4002fdaa22aa2, 0xbb51bb3ba4b3316a),
        (0xb7b1fe1d3a814203, 0xbb18a987773fb87c),
    ],
    [
        (0x3c62da0057f84d3c, 0x3fc70c511227d5aa),
        (0xb590c69cd835e65a, 0xb8fbdadd24cf38cb),
        (0xbc52da0057f84d3c, 0xbfb70c511227d5aa),
        (0x3bfe726f70245274, 0x3f5910ebe1f1cbcd),
        (0xbc0e61277dedf705, 0x3f7e7dc08e70e99a),
        (0x3bb740aaea2c85b5, 0xbf23bd7d159e09a5),
        (0x3bc9abf12a21a247, 0xbf300357a187375c),
        (0x3b7cab62f60660b1, 0x3ed5ec73302455b1),
        (0xbb76a66de44f6f2b, 0x3ed1e5d2836c8d99),
        (0xbad13f153900acfb, 0xbe78f41b94dfba1f),
        (0xbaff3652c8e5edf0, 0xbe68c11aada79ae8),
        (0xbaa209f480ef1062, 0x3e1141423452a28d),
        (0x3a9e19c8e523a095, 0x3df73ef6f4772772),
        (0xba405794c4ab8e4d, 0xbda007403efd2c9a),
        (0xb9c2a7b0a727a022, 0xbd7f93ab6667901d),
        (0xb9c5c7ae4a96c60d, 0x3d25638d0e0d9767),
        (0xb942fb1c3cc08579, 0x3d003d0592185f1c),
        (0x39373a63f1b5e963, 0xbca5832b6c129523),
        (0xb916b9966d8ed631, 0xbc7a2e88ff86af9c),
        (0x38c0ec56239fb4ce, 0x3c20e8839dc5443a),
        (0x3884c544300024fc, 0x3bf0ff71f08432b7),
        (0x3829d5c62bdc57b6, 0xbb955cc426feea1c),
        (0xb7ff81402cdf65ad, 0xbb6229c7418b10f0),
        (0x3773be52416792c2, 0x3b062f08720eaee7),
    ],
    [
        (0x35a8346aa8f5b4bb, 0xb909c0fc24f65a67),
        (0xbc6444d3d89ac00f, 0xbfc62d93aa9d05bb),
        (0xbc1afb8f729be80a, 0x3f70ba9ce88929f2),
        (0xbc2f64ee23828a95, 0x3f9d7073daebb02c),
        (0x3bd6ad884b57cbba, 0xbf462813c7f58733),
        (0xbbc1c5da610ae750, 0xbf574a948d05638e),
        (0x3ba7816485085533, 0x3f01695765f13bbf),
        (0x3b9157b692bbea45, 0x3f0170ab5eeb0e4e),
        (0x3b2715c0e1d13855, 0xbea9c8e418a20315),
        (0x3b394c7caa0cf226, 0xbe9e4d893dce335e),
        (0xbaee8ffa7c53b5fb, 0x3e460f73a6ccb16f),
        (0x3ad92747d94d94d7, 0x3e312782f0f2448a),
        (0x3a5745f5f5b1bc73, 0xbdd881d5842ec670),
        (0xba53516c45370c96, 0xbdbb4c521a316133),
        (0x39f845348a27f4b9, 0x3d63132da6b9ef80),
        (0xb9e1395c98d19f4a, 0x3d4019364d86610c),
        (0xb973bba720cb85bf, 0xbce5f4ab616b4a7d),
        (0xb92365a073b465df, 0xbcbd47ca99f425b7),
        (0xb90304c2a5a66771, 0x3c637371b17caa3a),
        (0xb8dc112a193fb5f1, 0x3c352a819018d0f2),
        (0x387a7ad3dc13170e, 0xbbdb5c2ea7518f33),
        (0xb848fe010506d44a, 0xbba8eaf2df7cede6),
        (0xb7e6b127ce015440, 0x3b4f4fdfa642b717),
        (0xb7a66fe7bcf681da, 0x3b185bcb7268f7c2),
    ],
    [
        (0x3c6a47ab4241a9f5, 0xbfc5664e13b70622),
        (0xb5ac924d5c123479, 0x3900f098dc38e480),
        (0xbc5a47ab4241a9f5, 0x3fb5664e13b70622),
        (0x3bddc134f1d74a51, 0xbf540ee3940b092f),
        (0xbbffe75afd6ce0fa, 0xbf7c5e1ad9fb2f40),
        (0xbbab866102da4707, 0x3f1fb8a98ef4a243),
        (0xbbc018bed7b77750, 0x3f2de9be58a373e2),
        (0xbb75089b2cae9b03, 0xbed1bec95415e630),
        (0x3b447f8eea3cc056, 0xbed0cf264341409e),
        (0x3ac71d5b65310e89, 0x3e74677d9e214cd4),
        (0x3b026d69949fca90, 0x3e6766cc63507104),
        (0xba902c01ede1dfd5, 0xbe0c905882b94f65),
        (0xba995672d0443f95, 0xbdf62157963573dc),
        (0xba18987abee2ab90, 0x3d9ae2e0cbb67b06),
        (0x3a01b88c86e07d4b, 0x3d7e456a1c8dbf40),
        (0xb9a82e3122996ca9, 0xbd222e4736637ff5),
        (0x397f896eb454c48a, 0xbcff56a09da19f70),
        (0xb939b4426eb8647c, 0x3ca28607cce512f7),
        (0x38da5f520ae3c80c, 0x3c796a2f3d059107),
        (0xb8bff36f14735622, 0xbc1d78e82e3dfe4c),
        (0xb8985995658ccf15, 0xbbf096501f12c820),
        (0x381ac1859ffca23b, 0x3b92d35996fa684b),
        (0xb808697949d6df60, 0x3b61ce65ead2f56a),
        (0x37a076885f254fcf, 0xbb03bf2c7d52312b),
    ],
    [
        (0x357834c50798eee0, 0xb9046916861c7cbe),
        (0xbc6e5d93454f99e3, 0x3fc4b2a2ebf61ece),
        (0x3bf37b02bd5f53a6, 0xbf6b3297fdae7902),
        (0x3c2eb354a6313bb4, 0xbf9b8105d59b114c),
        (0x3bb9bab6138b6903, 0x3f420a3f8c12a1ff),
        (0xbbf48ec1e2fa78e5, 0x3f55d18d69de6cfb),
        (0x3b0ad2f6c6600043, 0xbefc79db4b341e8e),
        (0x3b70c098a1ba083c, 0xbf00679c92c303b2),
        (0xbb4af505f8420dfd, 0x3ea53ac4ecd4be2d),
        (0x3b3bd880347c15c4, 0x3e9ca7507840c04b),
        (0xbadbc0ebcece96aa, 0xbe4253415f09767a),
        (0x3a8f2f3402a0f194, 0xbe305141505c9f00),
        (0xba73147caddad17a, 0x3dd4919940f05339),
        (0x3a20801c08cd53d5, 0x3dba20de95780e25),
        (0x3a0aad83cdc61600, 0xbd602fff3daabeb6),
        (0xb9cd95539d508286, 0xbd3f01d18581b85a),
        (0x3986a1dccaaee4ac, 0x3ce2d78e0d95c2b1),
        (0xb9420e56b2382383, 0x3cbc5cc76fc4031e),
        (0x390db41e1b715d4d, 0xbc60e00793400a38),
        (0xb8d97d5c56eaa5d3, 0xbc349c9cebd4542b),
        (0xb87767b6f54f3ae0, 0x3bd7fb7ba24468a5),
        (0x384f6e8dcd984b87, 0x3ba861a49c8b4594),
        (0xb7a3310c04ca52f6, 0xbb4bb5b1358f5d66),
        (0xb7bf2be7dc5aa7d5, 0xbb17ef6c51c02185),
    ],
    [
        (0x3c6316f8ffd294bc, 0x3fc40f90793605bb),
        (0xb57f9e2ffd50aa0c, 0xb8e84b55d671e3ab),
        (0xbc5316f8ffd294bc, 0xbfb40f90793605bb),
        (0x3bf7017919c133d4, 0x3f5085775a5529c9),
        (0xbbe51eb6f09db1a0, 0x3f7aa0ce0421d1a8),
        (0x3bb34ca5aae2a0a8, 0xbf1a32a28e4bc82e),
        (0x3b829f8de1c1e358, 0xbf2c26ebca0e46de),
        (0xbb655efa83f66d77, 0x3ecd7400876206d0),
        (0x3b4cb5b0088841fe, 0x3ecfc1bbf57e3ae2),
        (0xbb1b0133a27f94ef, 0xbe710c7090487d3e),
        (0x3aef5fb8bc32c77a, 0xbe6634db39e4a305),
        (0x3aa91e07a77165aa, 0x3e0810d7e4efab49),
        (0x3a839aaf703b3d25, 0x3df51b513b3c4cf1),
        (0x3a336ceec537a192, 0xbd96dd877beb3775),
        (0xba0a1b2154c0dad9, 0xbd7d06057acf98a1),
        (0xb9b49d50f1878b22, 0x3d1f3b007c53631a),
        (0x3998369e87f01d70, 0x3cfe341812329072),
        (0x394269c561f0abb0, 0xbca0119135f2618c),
        (0x390e03cff1d48846, 0xbc789daf7e05bf20),
        (0xb89d5c1dd0bd003d, 0x3c19d09c42c2fd20),
        (0xb89ec3091d0a20a5, 0x3bf023bdfeef2b62),
        (0x383eea157ba50462, 0xbb90a443b12390a2),
        (0x380faad5050249de, 0xbb6165e9e44cf4fc),
        (0x37ac0bb1b2874365, 0x3b019b4693645186),
    ],
    [
        (0xb563251dec0c8419, 0xb90190e21085c59e),
        (0x3c5948539688f9cf, 0xbfc37aac8c1aeabb),
        (0x3c0e1b9871576e7b, 0x3f66ac0d2e2f2f87),
        (0xbc3e2c9f8f0de524, 0x3f99e74e754ea71f),
        (0x3bc834f161e5b53a, 0xbf3e1c0589e32bb1),
        (0x3bfe8b15bd1b2805, 0xbf5496158dc5f7ff),
        (0x3b66422c8ac686b8, 0x3ef7d55405348ca9),
        (0x3b9138881216a9f6, 0x3eff0b30f4506228),
        (0xbb49c5af94afc68e, 0xbea1d9e3629b98ec),
        (0x3b31cbe80255f739, 0xbe9b35ef28e9de24),
        (0x3ac5335f73a99784, 0x3e3f0200e2650870),
        (0xbacc80c87bd0e106, 0x3e2f1fb225e849cb),
        (0xba76daf9021379d3, 0xbdd187928a64d042),
        (0xba42b8804ba3e9e1, 0xbdb908ba84482af1),
        (0xb9f798e4ea85d67c, 0x3d5bd0a97427e582),
        (0x39ae28268a102015, 0x3d3dd98a1888cb0d),
        (0x39800b32153c5fbf, 0xbce05368fdb3dbd3),
        (0x395973c2e28b46eb, 0xbcbb6e7eba27d4df),
        (0xb8fd64860bbe4d9f, 0x3c5d7e5c8f77a962),
        (0x388996f5c1af67cd, 0x3c34063c16fd02be),
        (0x386cbbfcf555cb4a, 0xbbd52213fb27642b),
        (0x384eb9fd51a68df4, 0xbba7c8fd7f555061),
        (0xb7c915bb96c79ac0, 0x3b489cfc43bd06a7),
        (0x37bcc11a3c54068e, 0x3b177056c238889b),
    ],
    [
        (0x3c689d1f48185c7e, 0xbfc2f2072e638cf4),
        (0xb5952cb9317551e1, 0x3901222bb44bcf14),
        (0xbc589d1f48185c7e, 0x3fb2f2072e638cf4),
        (0x3be64a78bbbb198f, 0xbf4bd42b64fc5bed),
        (0x3c082c4cf012e7aa, 0xbf792bb5e1e159fc),
        (0x3ba0f7052b5160b8, 0x3f161ace3386dfd7),
        (0xbbc911bb758a62b7, 0x3f2aa8d1cf8db852),
        (0x3b53e6d95cbd37f7, 0xbec8ef624c36fc32),
        (0x3b54f1d79d1c68cf, 0xbece26d3747fe829),
        (0x3b0b9f4f6af339a7, 0x3e6d010d2bdb6fa2),
        (0xbb08db6527b88cec, 0x3e65272828ae4057),
        (0x3a509976106e8349, 0xbe0497b03c4482e4),
        (0x3a79f5e122b84d8f, 0xbdf42e35495a0b9f),
        (0x39cbebe4937d13e3, 0x3d93b2e62efa9ef3),
        (0x3a14579198fd4bcf, 0x3d7bdc7867dae011),
        (0xb9bf0d7f81848e9e, 0xbd1b1acb454ca400),
        (0xb97ccb74999a40d8, 0xbcfd1ce7997b3c0d),
        (0xb9380511e3a7c101, 0x3c9c1b1fc8c0e04f),
        (0x38cac264c6c80d21, 0x3c77d2b971f6828b),
        (0xb8b8eacaf1c3a530, 0xbc16c0313e58129c),
        (0xb85cf03733411e25, 0xbbef5c5d17f54410),
        (0x3815be1ad7496a82, 0x3b8d8dcf1e84a9c2),
        (0xb7e5c044edc6060a, 0x3b60f6f0c4603224),
        (0x379126e0d0da66b9, 0xbaff7e6aff22be1e),
    ],
    [
        (0xb5627bd5768d6ae9, 0x38cc4a2cf6a18867),
        (0xbc389c717cff1eba, 0x3fc27407dfadee6d),
        (0x3c01b3998e2560e2, 0xbf6346950bfd91f1),
        (0x3c3dd532243e7d0c, 0xbf988d48d1d4eb7b),
        (0x3bcf03d11aeaafca, 0x3f399e6923ada922),
        (0x3bf2c9a985cf61c7, 0x3f538984b76cdf4a),
        (0x3b96fc0cb69470ac, 0xbef452194b75f3c7),
        (0xbb929abb7ce27bd1, 0xbefd855d7b8f0243),
        (0x3b33da9849a0f970, 0x3e9e8a88601ff296),
        (0x3aebb8f5815fd80b, 0x3e99f1bd69b16e6b),
        (0x3ac86d0d07634a8b, 0xbe3aa483fce054e3),
        (0x3ac5ab23f0a90bfb, 0xbe2dc4c02391c504),
        (0x3a69996a456ab65e, 0x3dce48dc1b4db792),
        (0x3a30cc58e232dc0c, 0x3db807112e6636fd),
        (0x39f6e9562163f106, 0xbd582c0e5f51b3f0),
        (0xb9c5c661501e7ee5, 0xbd3cc1592707b4bb),
        (0x3966aa764333f3df, 0x3cdc8f29e796ad34),
        (0xb95785ee30c62c1f, 0x3cba864ef1faff74),
        (0xb8ff8a02ad81ef19, 0xbc59f854cbd9efdc),
        (0x384627f385cebec8, 0xbc336efc940e4c98),
        (0xb87617b5b0e9289b, 0x3bd2bbd3c0ba4c14),
        (0xb84c5ed87745f0e5, 0x3ba72a3898c03cd9),
        (0x37e4b8d60a63fdb3, 0xbb45f681005965fc),
        (0xb7a7ad15bf5e48a5, 0xbb16e75915fef489),
    ],
    [
        (0x3c51f9b16832f362, 0x3fc1ff5eec6a01cd),
        (0xb59a05b7856c4a22, 0xb8ff07a73b062717),
        (0xbc41f9b16832f362, 0xbfb1ff5eec6a01cd),
        (0xbbe5f2ed43d9b503, 0x3f47daf64983af9d),
        (0xbc19a4b7b3ed5b9d, 0x3f77ed5fffc1c774),
        (0xbbafa9d6f1d224d2, 0xbf12f947962314a1),
        (0x3bba27855de20282, 0xbf296027ea1d6e5c),
        (0xbb585d6672f6864b, 0x3ec57486c67fbc78),
        (0x3b6dbae692eab5c3, 0x3eccc11a59e13739),
        (0x3b07ece30b8718ed, 0xbe690ade515567ae),
        (0x3afec36878fc6271, 0xbe6438a7e22c9734),
        (0x3aac5fba9cf90508, 0x3e01db6d29a7d048),
        (0x3a9c694efcaf05a9, 0x3df3588cd299009a),
        (0xba304f511d090519, 0xbd912b3d3f46a11d),
        (0xba1f96e81c9933ce, 0xbd7aca95934e0ad5),
        (0x39a4c6d8ef3baa67, 0x3d17c19e5c2a3a3a),
        (0xb98de02178f2e8a7, 0x3cfc15e96b25adba),
        (0x3904b1879dbd8e47, 0xbc98c7e06842ee3d),
        (0xb8f76e2dea2e96ba, 0xbc770ecfbf61ca0d),
        (0x38b68e1efbe2a988, 0x3c142e95d318b6f5),
        (0xb874c2ceb9ea4228, 0x3bee73932c419ab8),
        (0x37f46e3304b4ae0c, 0xbb8a60b7a857ccc0),
        (0x380c4d64ccf85fc7, 0xbb608626d8e04736),
        (0xb7990c5c39207c1f, 0x3afc47487352322d),
    ],
    [
        (0x35ab7cbed9b6b1f4, 0x390239f235357955),
        (0x3c6f5f4b08a76fd4, 0xbfc192f23ce3e051),
        (0x3bdee957b8603928, 0x3f60a668185c01b1),
        (0x3c39de5fb6906a4e, 0x3f9764141d652089),
        (0x3bd0a73bf1673e03, 0xbf3624437a2fe76a),
        (0xbbe29f0dad749d08, 0xbf52a184be0d9891),
        (0x3b92247582470aeb, 0x3ef196de0eeef190),
        (0x3b54bbca179d7e5e, 0x3efc317f854112ad),
        (0xbb3c4937a266ae2b, 0xbe9a8019ef772196),
        (0x3b1fa8fc16fd89b6, 0xbe98d38497beea33),
        (0x3ad1777abda5cdbe, 0x3e37318410813eeb),
        (0xbaa199670acf66d4, 0x3e2c8d9d45d76323),
        (0x3a61c9a46bc0a86d, 0xbdca782c1acc5d80),
        (0xba47414175e77281, 0xbdb71bbb74f830ea),
        (0x39fb21ba4fd0050e, 0x3d5539502b6b0d41),
        (0x39bde0e950bf1a4a, 0x3d3bbc22cd6ef8b2),
        (0xb951eab6630258a6, 0xbcd933a6402470d3),
        (0xb90f7f72f87b85f2, 0xbcb9a8f9bcb762d2),
        (0xb8f0fd489e307f2b, 0x3c5709caa1708994),
        (0xb8bf22055786ef23, 0x3c32db59b73d17f0),
        (0xb84510913d0bb141, 0xbbd0b5801d4ebd68),
        (0xb83e3fe31926f1ed, 0xbba68b53041d22be),
        (0xb7ecd50940caefad, 0x3b43b217fd3ffb23),
        (0x37a50bcc9d97c7ac, 0x3b165a991dfc9832),
    ],
    [
        (0x3c6e71c482be67bd, 0xbfc12dd57bf18ada),
        (0x35745b4d4ccbb68f, 0x38d7a991cf97ce2e),
        (0xbc5e71c482be67bd, 0x3fb12dd57bf18ada),
        (0x3bd11b1ac52dc1e2, 0xbf44bebeff7b7f02),
        (0xbc1286f932bea2b2, 0xbf76d9afe88301fa),
        (0x3b859ae0593310ce, 0x3f10842d50687949),
        (0xbbc5025e938d5cb6, 0x3f2841d86b9b92f4),
        (0x3b45bc90ed56ac34, 0xbec2b5caad1f2b9a),
        (0xbb3771ef99211bfa, 0xbecb86bad42fc220),
        (0x3ac3d72243d64739, 0x3e65e5117a965bcf),
        (0x3b07539fc5e6a80e, 0x3e6364a25cc7309d),
        (0x3a51d7761a4ff87d, 0xbdff53dcc9459e76),
        (0x3a99aab279c470e8, 0xbdf297f421bb27bc),
        (0xba27ed296e5c55e5, 0x3d8e3c9c7289c65b),
        (0xba1d2b7fc29cc174, 0x3d79cfbae2bdc016),
        (0xb99b7a6311b584d8, 0xbd1502858f3b1429),
        (0x399664d160d01e32, 0xbcfb20c42e642ccd),
        (0x391ef844bb66595d, 0x3c9604b99922c587),
        (0xb91aa631f65ad96a, 0x3c7654ba0a2b4ab4),
        (0xb8bee3929b6ac229, 0xbc120523ef69f26a),
        (0xb88fba8fecf60808, 0xbbed920e28727809),
        (0x3816d4cbcfecdce6, 0x3b87ab7af32311cf),
        (0xb7f2f0febd96b7f0, 0x3b60169fdd2f6938),
        (0x3782a08850768123, 0xbaf9808bbbe92c37),
    ],
    [
        (0x359dd81c30ee48fe, 0xb8f151a3aeac900c),
        (0x3c53f099a5f56db3, 0x3fc0cf3ed059c573),
        (0xbbff3cb9b807fae1, 0xbf5d242aa529931c),
        (0x3beb544962a1a242, 0xbf96613d93b0180b),
        (0x3bde039e229b4e70, 0x3f33627f261f5116),
        (0x3bf7fb39963ccff5, 0x3f51d69ca0d88394),
        (0xbb384fc5c48bb913, 0xbeeed574afab70fd),
        (0xbb95a44313a48eb8, 0xbefb06384e48ee5b),
        (0xbb387c094254f81d, 0x3e97452c65235728),
        (0xbaf881554ea8f625, 0x3e97d51f133b6843),
        (0xbad89e4d3b4b48e5, 0xbe346ac67a7e0c7c),
        (0x3ac74865f688f55b, 0xbe2b75d66a8d7fbb),
        (0x3a6411700cfe955c, 0x3dc75f5a5db478ef),
        (0xba50702b7e55e69b, 0x3db64531ca4ef464),
        (0x39b8d5aa5d138d2b, 0xbd52ceac14a86c69),
        (0xb9b1503306d5d1c3, 0xbd3aca497ff206bc),
        (0xb972644a7dcc9d17, 0x3cd66b96eb20b32f),
        (0xb954b544843ae3aa, 0x3cb8d889b2e2fb84),
        (0xb8e75b07f541d7cb, 0xbc5494ff17910999),
        (0x38c576f2b4ee6110, 0xbc324dc265486d0f),
        (0xb85a9820119b4156, 0x3bcdfc7ea51b3fad),
        (0x38430f69314493a8, 0x3ba5eff892c2d831),
        (0xb7dc83562058d69b, 0xbb41c095467f781e),
        (0x37bbc403c5629dc3, 0xbb15ce2e25be6e38),
    ],
    [
        (0x3c61a13e2fee5687, 0x3fc076826cc2c191),
        (0xb5391b2ea81aa5bf, 0x38c00dbc5df9a26a),
        (0xbc51a13e2fee5687, 0xbfb076826cc2c191),
        (0x3be79f62aa488a9d, 0x3f4241b03eaaf5d9),
        (0xbc15dbe9d7210c2e, 0x3f75e7f53001e4b1),
        (0x3ba5f4ab61cd1e51, 0xbf0d17978e2d0336),
        (0xbbb79a683ed1c80a, 0xbf2745b0df80666a),
        (0x3b63b64ff7d7f045, 0x3ec0803f7f7fe323),
        (0xbb50168184fb2a2f, 0x3eca7006e6ad9cfe),
        (0xbb09b47df06e299a, 0xbe63590d57d48525),
        (0xbad3c549213b58f2, 0xbe62a7084b42b890),
        (0x3a391d2842dc92cf, 0x3dfbc0dd5a22c9a6),
        (0xba9fb98847ec903b, 0x3df1e9e4e20477cd),
        (0xba2be602f82a15a9, 0xbd8ade1de2460eb2),
        (0x3a17881703d93302, 0xbd78ea3bc5e51ddd),
        (0x397ca6386538b34a, 0x3d12bb86a88ccf82),
        (0x399525cc1707dd4e, 0x3cfa3d6bcad0c3fc),
        (0x393d78e1fc33fbc8, 0xbc93b476152074ba),
        (0x39110f06e5116481, 0xbc75a59813c6ee3d),
        (0x38bc4c43696de6db, 0x3c1030bae23c4e59),
        (0xb844d08b5dab4a58, 0x3becba7872c6d35c),
        (0x37cead7255599d43, 0xbb855a77ed00896a),
        (0x37e2184f9f8c0595, 0xbb5f548425ac1d81),
        (0x3797f6a7c6e97b07, 0x3af71a13b7056576),
    ],
    [
        (0x357bf96ef321056c, 0x38e8be4b9ac053ef),
        (0x3c665439df5bb54c, 0xbfc0230b9797a7b3),
        (0x3bd7d914863ddca7, 0x3f59c8083b2b753a),
        (0xbc3c455b2fe151b8, 0x3f957d3203befd90),
        (0xbbd3de5214e17418, 0xbf3127cba22892de),
        (0x3bf0c5ff9d698a77, 0xbf51234471455a6c),
        (0xbb8a9c280a18d4c7, 0x3eeb4fe26ec3e489),
        (0xbb9cd17c659ae229, 0x3ef9fc5254f1086c),
        (0xbb3a46913af4f447, 0xbe94a44c6506c43f),
        (0x3b3f5055c4929bc5, 0xbe96f17dd184ad59),
        (0x3a9912b1a43a7c95, 0x3e3225640a6a9328),
        (0xbab880b364397a7e, 0x3e2a7943505d15ed),
        (0x3a4ecbcb0eff2849, 0xbdc4d296ecea7b61),
        (0x3a453c61fa23edd4, 0xbdb58177059b1ee7),
        (0x39e59e5790859f3e, 0x3d50cd71be5428ab),
        (0xb9dab853aa81b7a6, 0x3d39eafac486dfeb),
        (0x39594ed10e15fb80, 0xbcd417df1a94b2d2),
        (0xb940b5960ba1aada, 0xbcb81583d885c1c9),
        (0xb8fa5f714b1f1036, 0x3c52822e3f56f84b),
        (0xb8da3a40678675b0, 0x3c31c75f9fd23d85),
        (0x38595afb542f1092, 0xbbcb0fe75f5fce5e),
        (0xb8206d9263935574, 0xbba55a4472710434),
        (0x37b1baa0847432ef, 0x3b4014997c5c3405),
        (0xb7baecc415e584a1, 0x3b1544b15eb627ac),
    ],
    [
        (0x3c5d7cc4171715a0, 0xbfbfa8b41711c83a),
        (0x357640b3baebf195, 0xb8d6a301383ff88d),
        (0xbc4d7cc4171715a0, 0x3fafa8b41711c83a),
        (0x3bca510f3e4f0ba8, 0xbf403a8d0f110fe1),
        (0x3c195ccf34fc85bb, 0xbf7511c6dadaaa12),
        (0xbb6fad55e36d2414, 0x3f09e040fc62c87e),
        (0x3bb45c7350ce4d8a, 0x3f266582f66d8d4c),
        (0xbb2c8a29c78431da, 0xbebd62a18e287536),
        (0x3b667f7ad529fbab, 0xbec976fb023f0f79),
        (0x3b0f3a432d46b94e, 0x3e6141188eda6cd5),
        (0x3afac6e99d820d1b, 0x3e61fc77546c2a70),
        (0xba3e756c586379ea, 0xbdf8ccadf7842b28),
        (0xba8280b1979ddc28, 0xbdf14c0515097baa),
        (0x3a1187e8ad78c9df, 0x3d8810b7fe5b7aae),
        (0xba14e29f33da57b3, 0x3d78181478442ce9),
        (0xb9b79ebe6761a2d6, 0xbd10d30b1ed6e857),
        (0xb995f6bc31063165, 0xbcf96afe82155a3c),
        (0x3927bc733591bf77, 0x3c91c038cd6a8753),
        (0xb8f984f4b5077d41, 0x3c7501961b5bf1bf),
        (0xb8a4dc2a78afb902, 0xbc0d4372ee42b73d),
        (0xb88b4f0dd2f40637, 0xbbebee0a131ab576),
        (0xb81ffd82b16be530, 0x3b835cde101c242e),
        (0x37ec5511ff967fb0, 0x3b5e84462bfb49f0),
        (0x378914b866c350ff, 0xbaf50552417f971d),
    ],
    [
        (0xb556b1f952a5712e, 0xb8d6808df2a13f7c),
        (0x3c581bdf89b0a8b1, 0x3fbf13faf32c8e0a),
        (0xbbee91046256d1d1, 0xbf570558dddb7d46),
        (0xbc326d6d43908d3c, 0xbf94b24d7a933972),
        (0xbbb5423c2da87eef, 0x3f2ea52a21487a11),
        (0xbbf33353f135e755, 0x3f50834d8f3fdd5b),
        (0x3b8f2ed165228a36, 0xbee86941a4b43bea),
        (0xbb93d8441684ef00, 0xbef90e32cef3e900),
        (0xbb24c39a94ebc802, 0x3e92785f6385b273),
        (0xbaf58f1dccae728c, 0x3e962482bf9d2bb8),
        (0xbab1b25f4a923de6, 0xbe3043125386ac84),
        (0x3acbae29202faa10, 0xbe29943c303e31c2),
        (0xba689e50e7b79f94, 0x3dc2b2457a1921b4),
        (0xba5fc1c10645d26a, 0x3db4ce82afcbf544),
        (0x39e8d1c21f25be80, 0xbd4e3dcde42d462d),
        (0x39cd7cc147b1feaa, 0xbd391cdf33ca55c9),
        (0x396660b123953dac, 0x3cd220c06f5ae707),
        (0xb95a54474bf84f20, 0x3cb75f9d03ccf4de),
        (0x38e97ead5ea85772, 0xbc50be9744b83b69),
        (0x38d4e9dad41717fb, 0xbc3148975b0a87db),
        (0x38499800271bb705, 0x3bc88e2038a62c01),
        (0xb84c766c188201e5, 0x3ba4cb4d904c6a21),
        (0x37d9528d3a53acbd, 0xbb3d4587fbb07d3a),
        (0x37aa22e47790bf77, 0xbb14bfb26c12f06a),
    ],
    [
        (0x3c0020b4016594ac, 0x3fbe8727daa3daed),
        (0x357bd9f848327b9d, 0xb8d74664ec9bed14),
        (0xbbf020b4016594ac, 0xbfae8727daa3daed),
        (0xbbae628434d2d25b, 0x3f3d19c52e070d9f),
        (0xbc1361836c532514, 0x3f74524d4813cc25),
        (0xbb7cb5b9aae4885e, 0xbf0735f790b535f3),
        (0xbbcfed1fabeeb9ac, 0xbf259c8f9f0a3484),
        (0xbb45b8692da7d38c, 0x3eba619ffc5a3ad0),
        (0x3b69523f4f9b3609, 0x3ec896d7dc819faf),
        (0xbaac3c0f5c29d085, 0xbe5f04efbdfeacf4),
        (0x3afee241ecc24fbe, 0xbe6162253f3024f4),
        (0xba6d1ca080c948cb, 0x3df653d736c3ef75),
        (0xba92d3e20a44098b, 0x3df0bc406f716b40),
        (0x3a244bdd54062834, 0xbd85b5420be0cc44),
        (0x3a0a9968781f3a0f, 0xbd77573e801c99fe),
        (0x39a0139e52f2e139, 0x3d0e6b725d3b2ae6),
        (0xb99e56935f64b195, 0x3cf8a83e6e4c168c),
        (0x392ead92886f9ee6, 0xbc90163396acd7b1),
        (0x391d3d6b8efb3080, 0xbc74685be0d8a97e),
        (0xb8a6ef909392303e, 0x3c0a9718712afa89),
        (0x385c475ef4355891, 0x3beb2d18cd37a98f),
        (0x382d92b3e5d0a3a0, 0xbb81a4a7bc85c3b0),
        (0x37e8bb074772c480, 0xbb5dbd9c4b52231d),
        (0x3792acc4acbf596f, 0x3af335a8be4bc776),
    ],
    [
        (0x35982ed3998876be, 0xb8f5dc9ed9d5e051),
        (0xbc5a45a53b37a59e, 0xbfbe018d99f5da1b),
        (0x3be001e9c4dcf8a1, 0x3f54b85897b36265),
        (0xbc32db4ef3821944, 0x3f93fc442153435d),
        (0xbbcf201dea326b1a, 0xbf2b9694d71486e3),
        (0xbbef8e2381ba7f64, 0xbf4fe6fdc644ddde),
        (0x3b87d6d0f34bdbc3, 0x3ee5fd096e4523fb),
        (0xbb6b1063264f1631, 0x3ef83770c9a84498),
        (0x3b3a40ab832d99d2, 0xbe90a6f6f7e05f1b),
        (0xbb3f771b18195677, 0xbe956ad4a35eb0ad),
        (0x3ab57ad37dd35a78, 0x3e2d5bea474ca54f),
        (0xbac551a8f3695cff, 0x3e28c39f68d21b3c),
        (0x3a3d3669caabf1d6, 0xbdc0e73408ce7d88),
        (0xba59b3ea5bc5c592, 0xbdb42a6dedaa1a20),
        (0xb9ddcafa3dca5d8b, 0x3d4b640f953cb953),
        (0x397af903fd708234, 0x3d385e74fde5ce1a),
        (0x397d0253626bd0a6, 0xbcd0740945699eba),
        (0x39554011a8afef74, 0xbcb6b622427cf222),
        (0x38d2f100ca9a88c8, 0x3c4e76f4da90ff0f),
        (0x38d40ee48b131878, 0x3c30d15eeed4b2da),
        (0xb86bd3c37ebf7704, 0xbbc6647780d4dd5f),
        (0x384830339986f999, 0xbba44386be6e1434),
        (0x37b288dd01211d3e, 0x3b3ac32d5099a684),
        (0xb7bdc2d6cb33959c, 0x3b14400df422e81f),
    ],
    [
        (0xbc5cb1f28997ca39, 0xbfbd8293aa55d18f),
        (0x35ac5e4a18056845, 0x39005c1969d4e17b),
        (0x3c4cb1f28997ca39, 0x3fad8293aa55d18f),
        (0x3bb680f417abeace, 0xbf3a48fe4afedcc8),
        (0xbc16c091c5e2bd3b, 0xbf73a5ccbc12a67b),
        (0xbba33c90d1ef4886, 0x3f04f91e41eee9bc),
        (0xbb9b30870eff030b, 0x3f24e72224db2c0e),
        (0xbb169834dfa01495, 0xbeb7dac8202ad4fb),
        (0x3b6372510d6e3ed7, 0xbec7cbd49c315be0),
        (0x3aee3c01a205f644, 0x3e5c1396b62b0f84),
        (0x3af43728856e7e95, 0x3e60d5c64a9c427f),
        (0x3a8951d1fb56f9e5, 0xbdf43c4a5d5a74a1),
        (0x3a9dff3a9a1305b7, 0xbdf038cb3f5e324b),
        (0xba2227a6954a8107, 0x3d83b473da8d9c7c),
        (0x3a00f6b33cdc63fd, 0x3d76a5d59f9a81dd),
        (0xb988bc6d23c00f13, 0xbd0ba9a1977e4129),
        (0x398765f6255cb3d1, 0xbcf7f3d211d80ca6),
        (0x392695fb30ce2147, 0x3c8d5107593c868e),
        (0xb917c43ab17af1b0, 0x3c73d94e8dd594b7),
        (0x3898ca359e8db05d, 0xbc08484736827c1c),
        (0xb8762b3eb4479bca, 0xbbea7775ae8da32a),
        (0x37fbcf628aa86234, 0x3b80264bfaa20223),
        (0x37e271fa49b2756e, 0x3b5d00eb74cea235),
        (0x379f3f3a922be29c, 0xbaf1a063ed726ac9),
    ],
    [
        (0x35854b02b59f20a5, 0xb8e34bc17457894f),
        (0x3c5f215e77086bf5, 0x3fbd09b210b30217),
        (0x3bfc1f6fd559ad63, 0xbf52c74f6d120291),
        (0x3c0587bb3b3049f5, 0xbf9357bfc2be5860),
        (0xbbca26f6fb2c1060, 0x3f2901e4c495acea),
        (0x3bdfa22468c2245a, 0x3f4ee2a36979f905),
        (0xbb819f064e6a387e, 0xbee3f0cb93a497a4),
        (0x3b845c631a25decc, 0xbef7748921871bcc),
        (0x3af9ba7ee835a985, 0x3e8e39a085fc522a),
        (0x3b168e73edf85eb9, 0x3e94c1b7a6b2509f),
        (0xbacb21d04553bf28, 0xbe2aabe92ddd7d73),
        (0x3ac83c7d3d064b50, 0xbe2804c5ad3234f3),
        (0x3a2c4059ace3745e, 0x3dbec02b4af63bb7),
        (0x3a5e5f6d54d90428, 0x3db3938387c8ea5e),
        (0xb9a027aa900640db, 0xbd48f3df4c861285),
        (0xb9a31a1b83cb9055, 0xbd37ae3cd67c3467),
        (0x3962c6e3f82d4e4a, 0x3cce075e0968692d),
        (0x395097056b18178c, 0x3cb61833a4f480be),
        (0x38dbe00af812e106, 0xbc4bda71ed3079b1),
        (0xb8dbcc3ff8d7603a, 0xbc30616db3cfc278),
        (0x38543a30e58a4f05, 0x3bc483dc2d93ca5e),
        (0x383474a59c970c7b, 0x3ba3c2fe3717b507),
        (0xb7ce38a86dc4a340, 0xbb3892749970c097),
        (0xb7b9f0d1aaa91069, 0xbb13c62adb209305),
    ],
    [
        (0xbc49df1f0f8d2108, 0x3fbc96700bf039e2),
        (0xb59b9f4f3b516a10, 0xb900369fa71bf644),
        (0x3c39df1f0f8d2108, 0xbfac96700bf039e2),
        (0xbbb8cb9fe6e6c906, 0x3f37e5647d30fea8),
        (0x3c06397704521ddc, 0x3f73095734a24496),
        (0xbb56346f9ac2b6d1, 0xbf0312a4db537d5b),
        (0x3bc2ab35cecaca9f, 0xbf24424a96e62373),
        (0xbb5dc2fd60429a9d, 0x3eb5b4a6639fb7be),
        (0xbb533d879080abe9, 0x3ec712e4d44c4a74),
        (0xba8a3e2fc66990c5, 0xbe59917dedf003d7),
        (0x3b0b520e9010331b, 0xbe6055757b098917),
        (0x3a8b72c27ccfb937, 0x3df2728cdc02e18c),
        (0x3a7adc897bdb4467, 0x3def80393fe2fc80),
        (0xb9fad4f603343968, 0xbd81fcaa37a8b374),
        (0x3a03afc083aed050, 0xbd760225681cd60a),
        (0x39753518ad0e9b45, 0x3d0949ef9ee5d1b2),
        (0xb979524731c89f4a, 0x3cf74c654a2656ac),
        (0x391010895dc3b28c, 0xbc8ad88e518a28d9),
        (0xb9155499f6d680c6, 0xbc7353b6d93fd106),
        (0x38a0ae7b9e53be9c, 0x3c0646e795fc374a),
        (0x3872458a3118ee05, 0x3be9cca80299b7b1),
        (0xb81bebc11d5a928d, 0xbb7db0b7e5620486),
        (0xb7fe8030136a44be, 0xbb5c4e2a3f8819a4),
        (0x379be3a580646aab, 0x3af03c8d0fb251d3),
    ],
    [
        (0x359ca6ecae51f0ce, 0x38f785cd60d2c1ed),
        (0xbc37a2663626dcab, 0xbfbc28612a3bc18b),
        (0xbbf422f2d193784d, 0x3f511f52577ff6ba),
        (0x3c3976e0041aa23b, 0x3f92c21da135f56c),
        (0xbbc90eddb54be50c, 0xbf26ce18f8229e00),
        (0xbbed07e9f503136f, 0xbf4df586d8b786e1),
        (0xbb8f298c77e80033, 0x3ee230fede9c5ad4),
        (0xbb94e2ce3149cc01, 0x3ef6c2a7558fc928),
        (0xbb0453950e023ff5, 0xbe8b97329e667f58),
        (0x3b301d8210db0b8e, 0xbe9426ecfd66cd08),
        (0xbab764b9a347914a, 0x3e285e012a388a8c),
        (0x3aca4092574b7b84, 0x3e27557143798fcd),
        (0x3a3440e2c63fef00, 0xbdbc1f9f6fa068be),
        (0xba5fe0095953ab69, 0xbdb30842b25be60a),
        (0xb9d755f288da184b, 0x3d46d9757263a003),
        (0xb9bbfc5a456628aa, 0x3d370aced616e601),
        (0xb94bab13fb9e7f92, 0xbccb8986f8682a20),
        (0x3943b3e29fa75536, 0xbcb584e49cbf1ef2),
        (0x38ea081cc361bec5, 0x3c499540a7d48541),
        (0xb8c430149fc9b075, 0x3c2ff0b7143f7ffe),
        (0xb853ec9906091bb5, 0xbbc2e028bd3c8a70),
        (0xb83e36c66c5cd25b, 0xbba34986a6eaad8f),
        (0x37c533e5238e3cf1, 0x3b36a6785a72ecd5),
        (0x37a56b5d17159660, 0x3b1352244bd5f525),
    ],
    [
        (0x3c58fff4515190b5, 0xbfbbbf246914235f),
        (0xb538ee4bd006c414, 0xb8bef99392c87f53),
        (0xbc48fff4515190b5, 0x3fabbf246914235f),
        (0x3b9cd1da1498a8c8, 0xbf35d923e8470178),
        (0xbbfbef9e896a9a49, 0xbf727a96f174b6d1),
        (0xbb885ae82e4dfd87, 0x3f01715e4bbb00e7),
        (0xbbcb52802ba3fd33, 0x3f23abacdb5106b5),
        (0xbb5ee3261cd7bab4, 0xbeb3dc30d27849d7),
        (0x3b6dfe5ed028776c, 0xbec6698d6ee99eb9),
        (0x3aeb6c016c328002, 0x3e576911a4642dff),
        (0xbaedb58a3fa04f2e, 0x3e5fbf415682210b),
        (0x3a8ffce5bcffc5e8, 0xbdf0e7d3674631fb),
        (0x3a850b7e524d421f, 0xbdeea1cbf2d9d28f),
        (0x3a0972a562b54d3c, 0x3d80805ad6510626),
        (0x3a139fee7249181d, 0x3d756aaba545a661),
        (0xb994ba8b6414df04, 0xbd073ab529b32537),
        (0x39993a6f276892ca, 0xbcf6b0b8fe737496),
        (0x39299d1bde722fbd, 0x3c88b1cfe8649fd4),
        (0x391ed1ce51d7dd6c, 0x3c72d6d686eeb22f),
        (0xb8a7b2dc61e1a2b3, 0xbc0486115598b4ec),
        (0x38788c9ff45e0eaa, 0xbbe92c11b4ee051b),
        (0x380d5cd889980784, 0x3b7b6646fc1094e2),
        (0xb7d3fbaf9d9205e6, 0x3b5ba50d4ef914aa),
        (0x377a30dc81384d5d, 0xbaee055a0471899a),
    ],
    [
        (0xb51bc83c3a0e0aea, 0xb8afa7c9add8202e),
        (0x3c5d0edcbac85112, 0x3fbb5a6219b35e14),
        (0x3be54c0ba9407e4c, 0xbf4f645fdb1a8c89),
        (0x3c37a2e3fd65164b, 0xbf923940d01de8e9),
        (0x3bb0148b1871872e, 0x3f24e86a1e6384ff),
        (0xbbe17bbc96485963, 0x3f4d1c6a18c7ed95),
        (0xbb8c8ebe0fad0153, 0xbee0aeec61621923),
        (0xbb99391cbe507ac5, 0xbef61f7d30244338),
        (0xbaa992f34ae4f53d, 0x3e8950f781c1c41c),
        (0x3b3c536bd805608d, 0x3e93989942e07175),
        (0x3acb7edf5e20bddc, 0xbe265fd114ee2251),
        (0xbab668c775bebb6a, 0xbe26b3bd4dfd4b81),
        (0xba547f4392b0f26c, 0x3db9d8a1bbe00799),
        (0xba342c15e8cbe561, 0x3db2875c16e44880),
        (0xb9e8058b52e08699, 0xbd4505775bc7f118),
        (0x399839569d133457, 0xbd3672e2d4f2963b),
        (0x39448b9b5bdc1385, 0x3cc95d609dd374b0),
        (0x3922fcd538cfb114, 0x3cb4fb4d36b42032),
        (0xb8dba4f56475a1e4, 0xbc4798bab4a35bfe),
        (0xb8cbc77ae9cc73f8, 0xbc2f2b65eca95481),
        (0x38680bd8b9d2b683, 0x3bc16f8b2574d5ff),
        (0xb81bbb799c6ddbe9, 0x3ba2d6d14c18aef1),
        (0xb778386cee84a709, 0xbb34f49f286455da),
        (0x37858cd2c19af21f, 0xbb12e3e607d95973),
    ],
    [
        (0xbc5024304247ada3, 0x3fbaf9cb49c4f935),
        (0xb59f12f721a3d64f, 0x38fd67b1800cc2aa),
        (0x3c4024304247ada3, 0xbfaaf9cb49c4f935),
        (0xbbdc12274808bdaf, 0x3f3413b75ce0cc1b),
        (0x3c06b1ae600584f2, 0x3f71f7a8fec6eba8),
        (0xbbac4403ae7487a8, 0xbf0008442739ebfc),
        (0xbbc9d63344ad0f44, 0xbf23215dab7537c6),
        (0xbb3addfeaaf586b0, 0x3eb242e9e6bab199),
        (0xbb4ca6d60de07627, 0x3ec5cdc48f5d75eb),
        (0x3afbb75278987302, 0xbe5589cf32f43f3a),
        (0xbafb7ec07d34bce4, 0xbe5ee5f0d63c1125),
        (0x3a7c35d35717242c, 0x3def216eedb0f95e),
        (0x3a8f8e312b763d9b, 0x3dedd414a7d8c97f),
        (0x3a0cada00b33585e, 0xbd7e69fdff844763),
        (0x3a13060575a01497, 0xbd74de15be6889e6),
        (0x3953d8e70538cd11, 0x3d056e1418ad8cf1),
        (0x399662b0ae22d5ef, 0x3cf61fa985a16926),
        (0xb91ad31c7ecc6374, 0xbc86cf250611e72c),
        (0xb90005c6530d9799, 0xbc7261f414b70fcc),
        (0xb88ee36933bdb2e6, 0x3c02fb5ddae9e3c7),
        (0xb876a7ccb4b4f8e6, 0x3be895055837a30a),
        (0x37c443af13e1d435, 0xbb7960ab422d80c4),
        (0xb7d0c27f905f8955, 0xbb5b052425ccb1af),
        (0x37393e1ef7ae32d2, 0x3aebd91e8263d1bb),
    ],
    [
        (0xb5841211c9e4fbf8, 0x38e6aabd721697ea),
        (0x3c508b7cc7933a75, 0xbfba9d1835947d70),
        (0xbbcfb3814e43eef7, 0x3f4cea253049a85b),
        (0x3c3e993cf753c741, 0x3f91bb71f665dcdf),
        (0x3bc6de474edb7971, 0xbf23427f4797ae90),
        (0x3bab43046c577e07, 0xbf4c54a7bd6f30c1),
        (0xbb74fee4e6b715a1, 0x3edebe9e65809919),
        (0xbb914661eec5967d, 0x3ef58924fa089950),
        (0xbb1b1ce67962bafd, 0xbe87564501e7bcdc),
        (0xbb32d2083a6e1367, 0xbe9315306e5bb42b),
        (0x3abb19a5dd98b1c8, 0x3e24a32382dea2a5),
        (0xbacf847c762942f2, 0x3e261e0ee5367a22),
        (0xba5d25a1687ec831, 0xbdb7db8537582634),
        (0x3a40dc676669750a, 0xbdb20facb4a9bb8b),
        (0xb9e774de35526b14, 0x3d436bd6f0001ffb),
        (0x39c4a236b9d81483, 0x3d35e552618cbf26),
        (0x39679d0392cc254a, 0xbcc7754c8d46f92d),
        (0x394ad96fb5f59779, 0xbcb47a92b94007d3),
        (0xb8ef6b6af098cfd0, 0x3c45d920844ef181),
        (0xb8c4d1cfab1f2a0c, 0x3c2e71f74fd91a7e),
        (0xb86232b9682bb59c, 0xbbc02a09e0e77777),
        (0xb83a4f51c8d25c89, 0xbba26a7e606b3d3b),
        (0x37ac0d8d0af00df2, 0x3b33742f43880b52),
        (0x37b2cfe1ad3e0ef1, 0x3b127b3f3a3c24de),
    ],
    [
        (0xbc55d35a88f1e0a3, 0xbfba4407e04298d1),
        (0xb57a12f3e3faa5bd, 0xb8ebd844c546d1fe),
        (0x3c45d35a88f1e0a3, 0x3faa4407e04298d1),
        (0xbbdfcf49ead39f19, 0xbf3288694b34ab21),
        (0x3c1c3cb8ccc39cf1, 0xbf717f0266db2149),
        (0x3b7370deeb1fc99d, 0x3efd9a9a1d05433b),
        (0xbbc76ae488c15a1a, 0x3f22a1c916a2d5a1),
        (0xbb55855b88c468bb, 0xbeb0dd9e92661e07),
        (0xbb0d6efd0446126a, 0xbec53dd972d8f232),
        (0x3af0ee48a1c77965, 0x3e53e6e553aac740),
        (0xbadd40df09e40809, 0x3e5e1cc65033998a),
        (0xba32b59db0216721, 0xbdecc8a3f36b4e82),
        (0xba8b62a99a7fbe14, 0xbded152ceac3373c),
        (0x3a1c9ccca0c04104, 0x3d7c249e1f309fe3),
        (0x3a0ef225196657ba, 0x3d745b3c1804be30),
        (0xb9afb28e8d2ffa7d, 0xbd03d90bb0238840),
        (0xb97d594b3ede144b, 0xbcf598302ab3ac8a),
        (0xb92a26a4f451568c, 0x3c85259cd87cdd6c),
        (0xb912f6ad38a9745a, 0x3c71f460abad18ae),
        (0x388ed6abd853f84f, 0xbc019e5f6867ef21),
        (0xb8739b58b6a444ab, 0xbbe806d32d68c464),
        (0x380a7ea4b739cc38, 0x3b77958de1803c3e),
        (0x37ef37708fa55952, 0x3b5a6deb944c06b2),
        (0xb78f61f40b1c77dc, 0xbae9ea1590f503a9),
    ],
    [
        (0x357379a2bd7c52ce, 0xb8fbc8219db67734),
        (0x3c2748c82bad8c51, 0x3fb9ee5ee937fc89),
        (0x3bebaa9c093c19b6, 0xbf4abf28ad5bf6da),
        (0x3bf098a110e151ad, 0xbf9147481084ae0f),
        (0xbbcb58642b32be0e, 0x3f21d137345cfee5),
        (0x3bdbab7a25ecf8c2, 0x3f4b9c10ddf62464),
        (0x3b707cd222ddff83, 0xbedc72c9d49c6d7a),
        (0x3b9799bc03c5acb7, 0xbef4fe0b363ccc66),
        (0x3b2f6ffbbd8a09bd, 0x3e859a1c336b7506),
        (0xbb3843707d4c5068, 0x3e929b6632cc324d),
        (0xbaa20857a0292715, 0xbe231cd6e7991df9),
        (0xbab28b65ed5397d3, 0xbe25930872001f69),
        (0xba5cf09bb859c9ae, 0x3db61befbaafa006),
        (0xba2fa8cd2c2d9dc1, 0x3db1a038446ec46c),
        (0x39d429e8620217b4, 0xbd42030319ffc086),
        (0x39d5f3fcc28971f9, 0xbd35611796700e22),
        (0xb921b9f1f88c6b8d, 0x3cc5c664f879c556),
        (0x395775c1e2792ff0, 0x3cb401eb6a61fbc2),
        (0x38ead4ec597eb512, 0xbc444cf4a7922ede),
        (0x389608d2d57a3ecf, 0xbc2dc380c95ab35d),
        (0xb85138d52c6d34af, 0x3bbe1244da523859),
        (0x3811c5c33ebcb580, 0x3ba2042743d87f84),
        (0x37c0000bf1ca678b, 0xbb321df42759daa3),
        (0xb7b3e51de8fa1afc, 0xbb1217eed90481cc),
    ],
    [
        (0x3c5728ab934a26a0, 0x3fb99be744018c90),
        (0x359a217cfc0fd73f, 0x38f8cb0a4bb92b1c),
        (0xbc4728ab934a26a0, 0xbfa99be744018c90),
        (0xbbc84fc91475651a, 0x3f312d4e1c1ca4c3),
        (0x3c0f281d89ca12e4, 0x3f710f5ca51f98b0),
        (0x3b9fa3d9c46c2f87, 0xbefb71417476a570),
        (0xbb82784bcc79df7f, 0xbf222b9fa8dfd762),
        (0xbb2378ade6a74c3f, 0x3eaf471061b7dd7e),
        (0x3b688d4663776f74, 0x3ec4b862279de756),
        (0x3ae406b21b8c7a34, 0xbe52763f34a53566),
        (0xbafcf6291cf32841, 0xbe5d61e605e8e69a),
        (0x3a86957e31ae2fc6, 0x3deab7331a682502),
        (0x3a8a902f91e8c962, 0x3dec637428f3fda4),
        (0x39f24e732a100c1c, 0xbd7a236ad0b3eb49),
        (0xba105ad2fccf3d78, 0xbd73e11cef6eae93),
        (0xb995c95ce2902dca, 0x3d0272cbaef43092),
        (0xb995e32a69772db7, 0x3cf519623b4e50ac),
        (0xb9292fc5f6cbd421, 0xbc83ac6109d4a7c0),
        (0xb916701f83b25b26, 0xbc718d7ac32164f8),
        (0xb8a62d82d427d871, 0x3c0068369bb2c654),
        (0x388344bf0ff0ace9, 0x3be780d0943d0646),
        (0xb7f8d6f94baf1bf3, 0xbb75fc6a10345edb),
        (0xb7d82ddcd0a763ae, 0xbb59ded950660f9a),
        (0x378e6b4fc0afac52, 0x3ae82fa57e04df35),
    ],
    [
        (0xb59824ff875a6f4d, 0xb8fef5ad62580f34),
        (0xbc4e31356cad466d, 0xbfb94c6f54aef04b),
        (0xbbe68528f64c6571, 0x3f48d6371f018ef0),
        (0xbbf705a06948ad50, 0x3f90db975fd7dc47),
        (0x3bc06e812fbec708, 0xbf208bd1634353c9),
        (0x3beeb410edf15681, 0xbf4af0d3d4cd140d),
        (0x3b6d81ec6351ba6b, 0x3eda6c8e7c9d023e),
        (0xbb94f5f0c3884299, 0x3ef47cddcdac15d7),
        (0xbaee0980702c7029, 0xbe8412394fc50e0a),
        (0xbb3fb63544124d21, 0xbe922a21cb1edfe9),
        (0xbac10c16da7fd9d1, 0x3e21c417dedcf50f),
        (0xbaadc343b535f5ab, 0x3e25117f3061daaa),
        (0xba52f41f6480b3af, 0xbdb49009a6c8ef27),
        (0xba1394b790b42720, 0xbdb13823dda87c57),
        (0x39ecf2b99857d12c, 0x3d40c350b60502a2),
        (0x39d16c5f3587dedc, 0x3d34e54a89ae3a78),
        (0xb96e84c18fd85bb6, 0xbcc447dde87e69d6),
        (0xb92fa9fc1fe799a6, 0xbcb3909fa7a78345),
        (0x38e1d450f52eeca1, 0x3c42ec7e57d796d5),
        (0x38bc5414c0376e8d, 0x3c2d1f245967f392),
        (0x384f6a4fc5c93b04, 0xbbbc0ef954471f60),
        (0xb8436aa08a547d96, 0xbba1a364a86e2ed4),
        (0xb7c059ecba8db4c0, 0x3b30ebf4b4f1127f),
        (0xb792d76142a8ff7b, 0x3b11b9abb3510b7f),
    ],
    [
        (0x3c5e213a1a4b3671, 0xbfb8ffc9bd24fe08),
        (0x3568e6cd71e05784, 0xb8c846638d0f2565),
        (0xbc4e213a1a4b3671, 0x3fa8ffc9bd24fe08),
        (0xbbcd36327411471d, 0xbf2ff51b38eef42c),
        (0xbc0f532ddb23dab4, 0xbf70a7a725d3fbc4),
        (0xbb834bf088b48836, 0x3ef988128728122a),
        (0xbbb5bd608899de58, 0x3f21bdc846a09e43),
        (0x3b3013850114c517, 0xbead1b60a4812ac4),
        (0xbb5a77447ea6fe43, 0xbec43c2d8e698c10),
        (0x3ad8023dfb41774f, 0x3e512fd6ccdafecc),
        (0xbaf62e671fb10d65, 0x3e5cb3bb8802e835),
        (0xba859d9d3a171da7, 0xbde8e1dd09a92324),
        (0x3a8ce00fffe0e74e, 0xbdebbd84404b6237),
        (0x3a0700486612292d, 0x3d785bcfd7db3db0),
        (0x3a131220e196908b, 0x3d736ed760739d88),
        (0x399d6eab7135136a, 0xbd01343464751bc4),
        (0x399a9e6748bc0bea, 0xbcf4a26ef2219c2c),
        (0x3916fec76005e868, 0x3c825c3fff5cf7ca),
        (0xb8f4e1647a2b0045, 0x3c712caee4bc257a),
        (0xb895bdf1f04f1575, 0xbbfea67f8d5490ce),
        (0x388017eb7fef7e93, 0xbbe7025c0acb6ed2),
        (0xb80d402134b327c6, 0x3b748e326c33fa45),
        (0xb7f36f706b3d34ec, 0x3b595763182ec3fa),
        (0xb77ee821070f844b, 0xbae6a29de364d58e),
    ],
    [
        (0xb52b322a93406970, 0xb8e11567e5257506),
        (0x3c35e8738ef1b9ca, 0x3fb8b5ccad12d632),
        (0xbbd18507f867922a, 0xbf4724d0185980d2),
        (0x3c1f6ab95e49bd79, 0xbf907764ae2b1eeb),
        (0x3bbea971bec7b60e, 0x3f1ed6acc18fdf29),
        (0x3bca7600bd731e34, 0x3f4a51693df2efbb),
        (0x3b5424c5adeb27a5, 0xbed8a0ec61e4df7b),
        (0xbb2fcebf02eeb8d6, 0xbef4047f31fae045),
        (0xbb244d41e0ecfe16, 0x3e82b668af276950),
        (0x3b3cad4603e7028e, 0x3e91c07467ed6403),
        (0xbac9f09027e17c0c, 0xbe2091d32eea7f0f),
        (0xbacf61c0a2077065, 0xbe24987294de7d1e),
        (0x3a4c64cfb1cdc317, 0x3db32fe6ab220655),
        (0xba52d5ac79807dc5, 0x3db0d6b127f8ac22),
        (0x39d08853ae95a955, 0xbd3f4d16b8cb00ea),
        (0x39c9258c596bcdf2, 0xbd34711e3d50f99d),
        (0xb9607956efca23e8, 0x3cc2f28e5de9b115),
        (0x394e73cc5f03a293, 0x3cb326098eabb602),
        (0x38c9a844bf92f26d, 0xbc41b1699659661e),
        (0x38c79ee3937d0522, 0xbc2c84136a790cfd),
        (0xb84d287adf77e7fa, 0x3bba415dab11fbcc),
        (0x3843b689e4f1e579, 0x3ba147d23398faea),
        (0x37b913ca23ec4cc0, 0xbb2fb26f3c616548),
        (0x37a819aab969975c, 0xbb1160299cd9ebeb),
    ],
    [
        (0x3c5b1c9821974148, 0x3fb86e51be0a9153),
        (0xb5720ba7256e1fdc, 0x38ee1da34bfeb8e0),
        (0xbc4b1c9821974148, 0xbfa86e51be0a9153),
        (0xbbcc79e50a2dc677, 0x3f2dd3c244b53279),
        (0xbbe9cfc1363faa0e, 0x3f7046fc5a218a86),
        (0x3b47339108bed763, 0xbef7d51accbfba76),
        (0x3bbac27d1a98311f, 0xbf2157556c0e1b1a),
        (0xbb2e6885549f0df6, 0x3eab2d01c12c8227),
        (0xbb685c37a3fcd2ca, 0x3ec3c838897d0a1e),
        (0xbaf6ac2b47952135, 0xbe500d37aa536a8b),
        (0x3afe13ef178fd248, 0xbe5c10ecf088eac6),
        (0x3a640bc972bc618d, 0x3de73f8d1af57f4c),
        (0xba7d1502b5443bc9, 0x3deb2227fe94022c),
        (0xba0e6bfbee228ee6, 0xbd76c5390bcf017b),
        (0xba1975f49279a432, 0xbd7303a6ded9ee03),
        (0x399abf56810fc89c, 0x3d0017772c8c0084),
        (0x399af156baca0aee, 0x3cf4329cf32bcb6e),
        (0xb9219214145d8323, 0xbc812f535b46fd28),
        (0x3904bff1857123e5, 0xbc70d1776a208c18),
        (0x3888ba3ef05e5fd1, 0x3bfcb5a589ad0a5c),
        (0x3871815404beb4a9, 0x3be68adf143f5d78),
        (0xb8166a2869445d1c, 0xbb734508c6e2f203),
        (0x37edd1f20932a7cc, 0xbb58d702f0f1cb76),
        (0xb774cc6cd565f7bf, 0x3ae53cf6ad4248fb),
    ],
    [
        (0x3586adcfc24a15bd, 0x38fb4b40fa7659d2),
        (0x3c40ee4d5b1b8050, 0xbfb829356999a097),
        (0x3be3831592c6b168, 0x3f45a280e033eb59),
        (0x3c215c764657672c, 0x3f9019dba8336e48),
        (0x3b986e85a6375663, 0xbf1cd4559d95e619),
        (0xbbc0173a9412da83, 0xbf49bc857750335e),
        (0xbb63dae6774006f7, 0x3ed70706580693f9),
        (0x3b838496a72af1a4, 0x3ef393fc6cf8481f),
        (0xbaf56d91a42ef902, 0xbe81800af4665cb3),
        (0xbb2c21f37756b0a2, 0xbe915d91a2289a30),
        (0x3aa04b0681616769, 0x3e1f009c83572407),
        (0x3ab8eb4169a44614, 0x3e2427054565b344),
        (0xba4aede413980977, 0xbdb1f5167f464cc6),
        (0xba42c3010298ffea, 0xbdb07b3a2cb21729),
        (0xb9d5250b5c48f994, 0x3d3d4f453474d39c),
        (0x39d7de25bab32cb7, 0x3d3403dd864ce1b5),
        (0xb959c7b70a761729, 0xbcc1c096935c0515),
        (0x3949cfca62e35a95, 0xbcb2c193ec80c816),
        (0x38d72ce5fb9c0118, 0x3c40967d8ee82f54),
        (0x38b2d922c1971637, 0x3c2bf18fd1afe1de),
        (0x385409db53827004, 0xbbb8a21d0c689575),
        (0x384dc5f7e6e04311, 0xbba0f11083b9c847),
        (0x37ce8f2e696187cf, 0x3b2dc3272d23ffd6),
        (0xb7b0e86cfd0bfa4f, 0x3b110b1caef3d73d),
    ],
    [
        (0x3c21907f595a082a, 0xbfb7e656efb009ae),
        (0xb590e3abdd6e7404, 0x38f3c9c05fc68dd7),
        (0xbc11907f595a082a, 0x3fa7e656efb009ae),
        (0x3bcce5fd1ce6de53, 0xbf2bec6b33efcb49),
        (0x3bcf8f1c9736150f, 0xbf6fd932c26aad94),
        (0x3b80b8964c190cdc, 0x3ef6504d7e054d50),
        (0xbbb3c0e77f493518, 0x3f20f77ce6105150),
        (0x3b3d95fb56d1d86f, 0xbea972e573db9593),
        (0x3b44588b63a38eb3, 0xbec35ba58bf2f993),
        (0x3ad1cbaf2f9e7b25, 0x3e4e1246a53112c6),
        (0xbac8e1b1b14782dd, 0x3e5b7850b3bc8175),
        (0xba87c131e73bb3fd, 0xbde5c8dcec477a4e),
        (0xba84c1608c9f12fa, 0xbdea905348b4924d),
        (0x39e529efb683a910, 0x3d7558a1f12cdd3f),
        (0x3a18c3b5cbcd2fee, 0x3d729edf3b2865f3),
        (0x399e9dec38211b14, 0xbcfe2f9cdddbe89c),
        (0x3991f206b8c5d3a1, 0xbcf3c947b936c2fd),
        (0xb926d15a781493eb, 0x3c8020bb8706a05b),
        (0x391b89b4941b2fbc, 0x3c707b5bb2f1e971),
        (0x3875331d8b43e31b, 0xbbfaf6220e4137a4),
        (0x388fbf932efdfa5c, 0xbbe619cedd775137),
        (0xb7f15f7042e6e287, 0x3b721c04bebc0bb5),
        (0x37f3601b48c231f4, 0x3b585d398a9b6f0b),
        (0x378dd18a95cb152b, 0xbae3f99b23077622),
    ],
    [
        (0x352bf89ec4e05302, 0x38c2beead70ae7d0),
        (0xbc529270d2a71e91, 0x3fb7a597e9550934),
        (0xbbd42472048d4fe5, 0xbf44486c0b012534),
        (0xbc2ac52ac4327de6, 0xbf8f848eec0e0e6b),
        (0x3b684f34372ed852, 0x3f1b077fae06adc6),
        (0xbbce518d0a756962, 0x3f49310d6e6842c6),
        (0x3b230b8888fa2afa, 0xbed597a5befbd70e),
        (0x3b9e1e12047ef68f, 0xbef32a855e4e113b),
        (0xbb1d73ebaa41845b, 0x3e8069ba0b1a612d),
        (0xbb3a5058b2aec6be, 0x3e9100c97ba8e22d),
        (0xbab085381e3747da, 0xbe1d15b5350404fa),
        (0x3ac28b1288705592, 0xbe23bc775cca4413),
        (0x3a5ffd971088266f, 0x3db0da51f94c98f0),
        (0x3a5dfce6ace7ab08, 0x3db0252dc33a4d7a),
        (0xb9dc3fdc25aa9e9a, 0xbd3b84d751a36bd2),
        (0x39b47e26246286bb, 0xbd339ce8224cc6de),
        (0x395a80d541531404, 0x3cc0ad1ba9d01451),
        (0xb95b7f3b36aa04cc, 0x3cb262b8db934833),
        (0x38dc5e49fddd8fdf, 0xbc3f2ec77da02f0e),
        (0x38a348be97d223b5, 0xbc2b66ebaf597acc),
        (0x385369dda16c7b7d, 0x3bb72b1780503c41),
        (0x3840d18df6c3b214, 0x3ba09ec633447fba),
        (0xb7afbf9889f21003, 0xbb2c0312119c246e),
        (0xb7bc1892f4c2e089, 0xbb10ba3b452a59bb),
    ],
];
