/*
** (c) 1996-2000 The Regents of the University of California (through
** E.O. Lawrence Berkeley National Laboratory), subject to approval by
** the U.S. Department of Energy.  Your use of this software is under
** license -- the license agreement is attached and included in the
** directory as license.txt or you may contact Berkeley Lab's Technology
** Transfer Department at TTD@lbl.gov.  NOTICE OF U.S. GOVERNMENT RIGHTS.
** The Software was developed under funding from the U.S. Government
** which consequently retains certain rights as follows: the
** U.S. Government has been granted for itself and others acting on its
** behalf a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, and perform publicly
** and display publicly.  Beginning five (5) years after the date
** permission to assert copyright is obtained from the U.S. Department of
** Energy, and subject to any subsequent five (5) year renewals, the
** U.S. Government is granted for itself and others acting on its behalf
** a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, distribute copies to
** the public, perform publicly and display publicly, and to permit
** others to do so.
*/

#include "REAL.H"
#include "CONSTANTS.H"
#include "MACPROJ_F.H"
#include "BCTypes.H"

#define DIMS lo_1,lo_2,hi_1,hi_2
#define CDIMS loc_1,loc_2,hic_1,hic_2

c *************************************************************************
c ** INITSIGMA **
c ** Define the 1/rho coefficients at the top level of the multigrid
c *************************************************************************

      subroutine FORT_INITSIGMA(sigmax,sigmay,rho,areax,areay,DIMS,bc,ng)

      implicit none

      integer DIMS
      integer ng
      REAL_T sigmax(lo_1-ng+1:hi_1+ng  ,lo_2-ng+1:hi_2+ng-1)
      REAL_T sigmay(lo_1-ng+1:hi_1+ng-1,lo_2-ng+1:hi_2+ng  )
      REAL_T    rho(lo_1-3   :hi_1+3   ,lo_2-3   :hi_2+3)
      REAL_T  areax(lo_1     :hi_1+1   ,lo_2     :hi_2  )
      REAL_T  areay(lo_1     :hi_1     ,lo_2     :hi_2+1)
      integer bc(2,2)

c     Local variables
      integer i,j

      do j = lo_2,hi_2 
        do i = lo_1,hi_1+1 
          sigmax(i,j) = two*areax(i,j)/(rho(i,j) + rho(i-1,j))
        enddo

        if (BCX_LO .eq. WALL .or. BCX_LO .eq. INLET) then
          sigmax(lo_1,j) = areax(lo_1,j) / rho(lo_1-1,j)
        else if (BCX_LO .eq. OUTLET) then
          sigmax(lo_1,j) = areax(lo_1,j) / rho(lo_1,j)
        endif

        if (BCX_HI .eq. WALL .or. BCX_HI .eq. INLET) then
          sigmax(hi_1+1,j) = areax(hi_1+1,j) / rho(hi_1+1,j)
        else if (BCX_HI .eq. OUTLET) then
          sigmax(hi_1+1,j) = areax(hi_1+1,j) / rho(hi_1,j)
        endif
      enddo

      do i = lo_1,hi_1 
        do j = lo_2,hi_2+1 
          sigmay(i,j) = two * areay(i,j) /
     $                  (rho(i,j) + rho(i,j-1))
        enddo

        if (BCY_LO .eq. WALL .or. BCY_LO .eq. INLET) then
          sigmay(i,lo_2) = areay(i,lo_2) / rho(i,lo_2-1)
        else if (BCY_LO .eq. OUTLET) then
          sigmay(i,lo_2) = areay(i,lo_2) / rho(i,lo_2)
        endif

        if (BCY_HI .eq. WALL .or. BCY_HI .eq. INLET) then
          sigmay(i,hi_2+1) = areay(i,hi_2+1) / rho(i,hi_2+1)
        else if (BCY_HI .eq. OUTLET) then
          sigmay(i,hi_2+1) = areay(i,hi_2+1) / rho(i,hi_2)
        endif
      enddo

      return
      end

c *************************************************************************
c ** PROJUMAC **
c ** Update the edge-based velocities
c *************************************************************************

      subroutine FORT_PROJUMAC(uadv,vadv,sigmax,sigmay,phi,
     $                         areax,areay,dx,DIMS,bc,is_rz,ng)

      implicit none

      integer DIMS
      integer ng
      REAL_T   uadv(lo_1  :hi_1+1,lo_2  :hi_2  )
      REAL_T   vadv(lo_1  :hi_1  ,lo_2  :hi_2+1)
      REAL_T sigmax(lo_1-ng+1:hi_1  +ng,lo_2-ng+1:hi_2-1+ng)
      REAL_T sigmay(lo_1-ng+1:hi_1-1+ng,lo_2-ng+1:hi_2  +ng)
      REAL_T    phi(lo_1-ng:hi_1  +ng,lo_2-ng:hi_2  +ng)
      REAL_T  areax(lo_1  :hi_1+1,lo_2  :hi_2  )
      REAL_T  areay(lo_1  :hi_1  ,lo_2  :hi_2+1)
      REAL_T     dx(2)
      integer is_rz
      integer bc(2,2)

c     Local variables
      integer i,j
      integer istart

      call setbc(phi,DIMS,bc,ng)

      istart = cvmgt(lo_1+1,lo_1,is_rz .eq. 1 .and. lo_1 .eq. 0)

      do j = lo_2,hi_2
        do i = istart,hi_1+1
          uadv(i,j) = uadv(i,j) - sigmax(i,j) / areax(i,j) * 
     $                (phi(i,j)-phi(i-1,j))/dx(1)
        enddo
      enddo

      do j = lo_2,hi_2+1
        do i = lo_1,hi_1
          vadv(i,j) = vadv(i,j) - sigmay(i,j) / areay(i,j) * 
     $                (phi(i,j)-phi(i,j-1))/dx(2)
        enddo
      enddo

      return
      end

c *************************************************************************
c ** RESIDUAL **
c ** Compute the residual R = f - D( sig G(phi) )
c *************************************************************************

      subroutine FORT_RESIDUAL(resid,phi,f,sigmax,sigmay,DIMS,dx,resnorm,bc,ng)

      implicit none

      integer DIMS
      integer ng
      REAL_T  resid(lo_1-ng+1:hi_1+ng-1,lo_2-ng+1:hi_2+ng-1)
      REAL_T    phi(lo_1-ng  :hi_1+ng  ,lo_2-ng  :hi_2+ng  )
      REAL_T      f(lo_1-ng+1:hi_1+ng-1,lo_2-ng+1:hi_2+ng-1)
      REAL_T sigmax(lo_1-ng+1:hi_1  +ng,lo_2-ng+1:hi_2-1+ng)
      REAL_T sigmay(lo_1-ng+1:hi_1-1+ng,lo_2-ng+1:hi_2  +ng)
      REAL_T dx(2)
      REAL_T resnorm
      integer bc(2,2)

c     Local variables
      REAL_T hxinv, hyinv
      REAL_T rfac, corr
      integer i,j

      hxinv = one/dx(1)
      hyinv = one/dx(2)

      resnorm = zero

      call setbc(phi,DIMS,bc,ng)

      do j = lo_2,hi_2 
        do i = lo_1,hi_1 

          rfac = ((sigmax(i+1,j) + sigmax(i,j))*hxinv + 
     $            (sigmay(i,j+1) + sigmay(i,j))*hyinv )
          corr =  
     $      ( sigmax(i+1,j)*phi(i+1,j) + sigmax(i,j)*phi(i-1,j))*hxinv + 
     $      ( sigmay(i,j+1)*phi(i,j+1) + sigmay(i,j)*phi(i,j-1))*hyinv 

          resid(i,j) = f(i,j) - (corr - rfac*phi(i,j))

          resnorm = max(resnorm, abs(resid(i,j)))

        enddo
      enddo

      return
      end

c *************************************************************************
c ** RELAX **
c ** Gauss-Seidel or line solve relaxation (depending on hy/hx)
c *************************************************************************

      subroutine FORT_RELAX(phi,f,sigmax,sigmay,
     $                      DIMS,dx,bc,nnrelax,ng)

      implicit none

      integer DIMS
      integer ng
      REAL_T    phi(lo_1-ng  :hi_1+ng  ,lo_2-ng  :hi_2+ng  )
      REAL_T      f(lo_1-ng+1:hi_1+ng-1,lo_2-ng+1:hi_2+ng-1)
      REAL_T sigmax(lo_1-ng+1:hi_1  +ng,lo_2-ng+1:hi_2-1+ng)
      REAL_T sigmay(lo_1-ng+1:hi_1-1+ng,lo_2-ng+1:hi_2  +ng)
      REAL_T dx(2)
      integer bc(2,2)
      integer nnrelax

c     Local variables
      REAL_T rfac, corr
      integer i, j, iinc
      integer iter
      integer extra_xlo, extra_ylo
      integer extra_xhi, extra_yhi

      if (ng .lt. 2*nnrelax) then
        print *,'NOT ENOUGH GHOST CELLS FOR RELAX '
        print *,'NG ',ng
        print *,'NNRELAX ',nnrelax
        stop
      endif

      call setbc(phi,DIMS,bc,ng)

      do iter = 1, 2*nnrelax

          extra_xlo = cvmgt(ng-iter,0,BCX_LO .eq. INTERIOR .or. BCX_LO .eq. PERIODIC)
          extra_xhi = cvmgt(ng-iter,0,BCX_HI .eq. INTERIOR .or. BCX_HI .eq. PERIODIC)
          extra_ylo = cvmgt(ng-iter,0,BCY_LO .eq. INTERIOR .or. BCY_LO .eq. PERIODIC)
          extra_yhi = cvmgt(ng-iter,0,BCY_HI .eq. INTERIOR .or. BCY_HI .eq. PERIODIC)

          do j = lo_2-extra_ylo,hi_2+extra_yhi 
           iinc = mod(j+iter+1+extra_xlo+2*ng,2)
           do i = lo_1-extra_xlo+iinc,hi_1+extra_xhi,2

              rfac = ( (sigmax(i+1,j) + sigmax(i,j))/dx(1) +
     $                 (sigmay(i,j+1) + sigmay(i,j))/dx(2) )

              corr =
     $        ( sigmax(i+1,j)*phi(i+1,j) + sigmax(i,j)*phi(i-1,j))/dx(1) +
     $        ( sigmay(i,j+1)*phi(i,j+1) + sigmay(i,j)*phi(i,j-1))/dx(2)

              rfac = one / rfac

              phi(i,j) = (corr - f(i,j)) * rfac

           enddo
         enddo

         call setbc(phi,DIMS,bc,ng)

      enddo

      return
      end


c *************************************************************************
c ** BC **
c ** Impose boundary conditions
c *************************************************************************

      subroutine setbc(phi,DIMS,bc,ng)

      implicit none

      integer DIMS
      integer ng
      REAL_T  phi(lo_1-ng:hi_1+ng,lo_2-ng:hi_2+ng)
      integer bc(2,2)

c     Local variables
      integer i, j, is, ie, js, je

      is = lo_1
      ie = hi_1
      js = lo_2
      je = hi_2

      if (BCY_LO .eq. OUTLET) then
        do i = is-ng,ie+ng 
          phi(i,js-1) = -phi(i,js)
        enddo
      elseif (BCY_LO .eq. INLET .or. BCY_LO .eq. WALL) then
        do i = is-ng,ie+ng 
          phi(i,js-1) =  phi(i,js)
        enddo
      endif

      if (BCY_HI .eq. OUTLET) then
        do i = is-ng,ie+ng 
          phi(i,je+1) = -phi(i,je)
        enddo
      elseif (BCY_HI .eq. INLET .or. BCY_HI .eq. WALL) then
        do i = is-ng,ie+ng 
          phi(i,je+1) = phi(i,je)
        enddo
      endif

      if (BCX_LO .eq. OUTLET) then
        do j = js-ng,je+ng 
          phi(is-1,j) = -phi(is,j)
        enddo
      elseif (BCX_LO .eq. INLET .or. BCX_LO .eq. WALL) then
        do j = js-ng,je+ng 
          phi(is-1,j) =  phi(is,j)
        enddo
      endif

      if (BCX_HI .eq. OUTLET) then
        do j = js-ng,je+ng 
          phi(ie+1,j) = -phi(ie,j)
        enddo
      elseif (BCX_HI .eq. INLET .or. BCX_HI .eq. WALL) then
        do j = js-ng,je+ng 
          phi(ie+1,j) = phi(ie,j)
        enddo
      endif

      return
      end

c *************************************************************************
c ** RHSMAC **
c ** Compute the right-hand-side D(U) for the MAC projection
c *************************************************************************

      subroutine FORT_RHSMAC(uadv,vadv,divu_src,rhs,
     $                       areax,areay,vol,DIMS,rhsnorm,ng)

      implicit none

      integer DIMS
      integer ng
      REAL_T     uadv(lo_1   :hi_1+1 ,lo_2   :hi_2   )
      REAL_T     vadv(lo_1   :hi_1   ,lo_2   :hi_2+1 )
      REAL_T divu_src(lo_1   :hi_1   ,lo_2   :hi_2   )
      REAL_T      rhs(lo_1-ng+1:hi_1+ng-1,lo_2-ng+1:hi_2+ng-1)
      REAL_T    areax(lo_1   :hi_1+1 ,lo_2   :hi_2   )
      REAL_T    areay(lo_1   :hi_1   ,lo_2   :hi_2+1 )
      REAL_T      vol(lo_1-1 :hi_1+1 ,lo_2-1 :hi_2+1 )
      REAL_T  rhsnorm

c     Local variables
      integer i,j

      rhsnorm = zero
      do j = lo_2,hi_2
        do i = lo_1,hi_1

          rhs(i,j) = areax(i+1,j)*uadv(i+1,j) - areax(i,j)*uadv(i,j) +
     $               areay(i,j+1)*vadv(i,j+1) - areay(i,j)*vadv(i,j)

          rhs(i,j) = rhs(i,j) - vol(i,j)*divu_src(i,j)

          rhsnorm = max(rhsnorm,abs(rhs(i,j)))

        enddo
      enddo

      return
      end

c *************************************************************************
c ** COARSIGMA **
c ** Coarsen the edge-based sigma coefficients
c *************************************************************************

      subroutine FORT_COARSIGMA(sigmax,sigmay,sigmaxc,sigmayc,DIMS,CDIMS,ng)

      implicit none
      integer DIMS
      integer CDIMS
      integer ng
      REAL_T  sigmax(lo_1 -ng+1:hi_1   +ng,lo_2 -ng+1:hi_2 -1+ng)
      REAL_T  sigmay(lo_1 -ng+1:hi_1 -1+ng,lo_2 -ng+1:hi_2   +ng)
      REAL_T sigmaxc(loc_1-ng+1:hic_1  +ng,loc_2-ng+1:hic_2-1+ng)
      REAL_T sigmayc(loc_1-ng+1:hic_1-1+ng,loc_2-ng+1:hic_2  +ng)

c     Local variables
      integer i,j

      do j = lo_2,hi_2,2
        do i = lo_1,hi_1+1,2
          sigmaxc(i/2,j/2) = (sigmax(i,j) + sigmax(i,j+1)) * fourth
        enddo
      enddo

      do j = lo_2,hi_2+1,2
        do i = lo_1,hi_1,2
          sigmayc(i/2,j/2) = (sigmay(i,j) + sigmay(i+1,j)) * fourth
        enddo
      enddo

      return
      end

c *************************************************************************
c ** RESTRICT **
c ** Conservatively average the residual
c *************************************************************************

      subroutine FORT_RESTRICT(res,resc,DIMS,CDIMS,nextra)

      implicit none
      integer DIMS
      integer CDIMS
      integer nextra
      REAL_T  res(lo_1 -nextra:hi_1 +nextra,lo_2 -nextra:hi_2 +nextra)
      REAL_T resc(loc_1-nextra:hic_1+nextra,loc_2-nextra:hic_2+nextra)

c     Local variables
      integer i,j

      do j = lo_2,hi_2,2
        do i = lo_1,hi_1,2
          resc(i/2,j/2) =  (res(i  ,j) + res(i  ,j+1) +
     $                      res(i+1,j) + res(i+1,j+1) ) * fourth

        enddo
      enddo

      return
      end

c *************************************************************************
c ** INTERPOLATE **
c ** Piecewise constant interpolation
c *************************************************************************

      subroutine FORT_INTERPOLATE(phi,deltac,DIMS,CDIMS,ng)

      implicit none
      integer DIMS
      integer CDIMS
      integer ng
      REAL_T    phi(lo_1 -ng:hi_1 +ng,lo_2 -ng:hi_2 +ng)
      REAL_T deltac(loc_1-ng:hic_1+ng,loc_2-ng:hic_2+ng)

c     Local variables
      integer i,j

      do j = lo_2,hi_2
        do i = lo_1,hi_1
          phi(i,j) = phi(i,j) + deltac(i/2,j/2)
        enddo
      enddo

      return
      end

c *************************************************************************
c ** SOLVEMAC **
c ** Conjugate gradient bottom-solver
c *************************************************************************

      subroutine FORT_SOLVEMAC(dest, dest0, source, sigmax, sigmay, sum, 
     $                         r, w, z, work, DIMS, dx, bc, norm, prob_norm, ng)

      implicit none

      integer DIMS
      integer ng
      REAL_T   dest(lo_1-ng:hi_1+ng,lo_2-ng:hi_2+ng)
      REAL_T  dest0(lo_1-1 :hi_1+ 1,lo_2-1 :hi_2+1)
      REAL_T source(lo_1-ng+1:hi_1+ng-1,lo_2-ng+1:hi_2+ng-1)
      REAL_T sigmax(lo_1-ng+1:hi_1  +ng,lo_2-ng+1:hi_2-1+ng)
      REAL_T sigmay(lo_1-ng+1:hi_1-1+ng,lo_2-ng+1:hi_2  +ng)
      REAL_T    sum(lo_1-1:hi_1+1,lo_2-1:hi_2+1)
      REAL_T      r(lo_1-1:hi_1+1,lo_2-1:hi_2+1)
      REAL_T      w(lo_1-1:hi_1+1,lo_2-1:hi_2+1)
      REAL_T      z(lo_1-1:hi_1+1,lo_2-1:hi_2+1)
      REAL_T   work(lo_1-1:hi_1+1,lo_2-1:hi_2+1)
      REAL_T dx(2)
      integer bc(2,2)
      REAL_T norm
      REAL_T prob_norm

c     Local variables
      integer i,j,iter,is,ie,js,je
      REAL_T alpha, beta, rho, rhol
      REAL_T  tol,tolfac
      REAL_T local_norm
      REAL_T sum_src
      logical is_singular

      tolfac = 1.0d-3

      is = lo_1
      js = lo_2
      ie = hi_1
      je = hi_2

      call macperiodic(dest,DIMS,bc,ng)
      call       setbc(dest,DIMS,bc,ng)

      do j = js-1,je+1 
        do i = is-1,ie+1 
          dest0(i,j) = dest(i,j)
           dest(i,j) = zero
        enddo
      enddo

      is_singular = (BCX_LO .ne. OUTLET .and. BCX_HI .ne. OUTLET .and.
     $               BCY_LO .ne. OUTLET .and. BCY_HI .ne. OUTLET)

      sum_src = zero
      if (is_singular) then
        do j = js, je 
        do i = is, ie 
          sum_src = sum_src + source(i,j)
        enddo
        enddo
        sum_src = sum_src / float((je-js+1)*(ie-is+1))
        do j = js, je 
        do i = is, ie 
          source(i,j) = source(i,j) - sum_src
        enddo
        enddo
      endif


 10   do j = js,je 
        do i = is,ie 
          w(i,j) = 
     $     ( sigmax(i+1,j)*dest0(i+1,j) + 
     $       sigmax(i  ,j)*dest0(i-1,j) )/dx(1)
     $    +( sigmay(i,j+1)*dest0(i,j+1) + 
     $       sigmay(i,j  )*dest0(i,j-1) )/dx(2)
     $    -((sigmax(i+1,j) + sigmax(i,j))/dx(1)
     $     +(sigmay(i,j+1) + sigmay(i,j))/dx(2) ) * dest0(i,j)
        enddo
      enddo

      do j = js, je 
        do i = is, ie 
          r(i,j) = source(i,j) - w(i,j)
        enddo
      enddo

      rho        = zero
      norm       = zero
      local_norm = zero

      do j = js, je 
        do i = is, ie 
          z(i,j) = r(i,j) 
          rho = rho + z(i,j) * r(i,j)
          local_norm = max(local_norm,abs(r(i,j)))
          norm       = max(norm      ,abs(r(i,j)))
        enddo
      enddo

      tol = Max(tolfac*local_norm,1.0d-15*prob_norm)

      if (norm .le. tol) then
        if (tolfac .gt. 1.0d-3) then
          do j = js-1,je+1 
          do i = is-1,ie+1 
            dest(i,j) = zero
          enddo
          enddo
        endif
        return
      endif

      do j = js, je 
        do i = is, ie 
          work(i,j) = zero
          dest(i,j) = z(i,j)
        enddo
      enddo

      iter = 0

c     write(6,1000) iter, norm/prob_norm

 100  call macperiodic(dest,DIMS,bc,ng)
      call       setbc(dest,DIMS,bc,ng)

      do j = js,je 
        do i = is,ie 
          w(i,j) = 
     $     ( sigmax(i+1,j)*dest(i+1,j) + 
     $       sigmax(i  ,j)*dest(i-1,j) )/dx(1)
     $    +( sigmay(i,j+1)*dest(i,j+1) + 
     $       sigmay(i,j  )*dest(i,j-1) )/dx(2)
     $    -((sigmax(i+1,j) + sigmax(i,j))/dx(1) + 
     $      (sigmay(i,j+1) + sigmay(i,j))/dx(2) ) * dest(i,j)
        enddo
      enddo


      alpha = zero
      do j = js, je 
        do i = is, ie 
          alpha = alpha + dest(i,j)*w(i,j)
        enddo
      enddo

      alpha = rho / alpha
      rhol = rho
      rho = zero
      norm = zero

      do j = js, je 
        do i = is, ie 
          work(i,j) = work(i,j) + alpha * dest(i,j)
          r(i,j) = r(i,j) - alpha * w(i,j)
          z(i,j) = r(i,j)
          rho = rho + z(i,j) * r(i,j)
          norm = max(norm,abs(r(i,j)))
        enddo
      enddo

      iter = iter+1
c     write(6,1000) iter, norm/prob_norm

      if (norm .le. tol) then
         
         do j = js, je 
            do i = is, ie 
               dest(i,j) = work(i,j) + dest0(i,j)
            enddo
         enddo

         call setbc(dest,DIMS,bc,ng)

      else if (iter .ge. 100  .or.  norm .ge. 100.d0 * local_norm) then

         tolfac = 10.d0 * tolfac
         iter = 1
         do j = js, je 
            do i = is, ie 
               dest(i,j) = work(i,j) + dest0(i,j)
            enddo
         enddo
         goto 10

      else

        beta = rho / rhol
        do j = js, je 
          do i = is, ie 
            dest(i,j) = z(i,j) + beta * dest(i,j)
          enddo
        enddo
        goto 100
      endif

c      call flush(6)

1000  format('Res/Res0 in solve: ',i4,2x,e12.5)
c     call flush(6)

      return
      end

c *************************************************************************
c ** MKSUMMAC **
c ** Pre-compute the sum of coefficients for the conjugate gradient solver
c *************************************************************************

      subroutine FORT_MKSUMMAC(sum,sigmax,sigmay,DIMS,dx,ng)

      implicit none

      integer DIMS
      integer ng
      REAL_T    sum(lo_1-1   :hi_1+1   ,lo_2-1   :hi_2+1   )
      REAL_T sigmax(lo_1-ng+1:hi_1  +ng,lo_2-ng+1:hi_2-1+ng)
      REAL_T sigmay(lo_1-ng+1:hi_1-1+ng,lo_2-ng+1:hi_2  +ng)
      REAL_T dx(2)

c     Local variables
      integer i,j

      do j = lo_2,hi_2
        do i = lo_1,hi_1
          sum(i,j) = ( (sigmax(i+1,j) + sigmax(i,j))/dx(1)
     $                +(sigmay(i,j+1) + sigmay(i,j))/dx(2) )
          sum(i,j) = -fourth*sum(i,j)
        enddo
      enddo

      return
      end



c *************************************************************************
c ** TRIDIAG **
c ** Do a tridiagonal solve 
c *************************************************************************

      subroutine tridiag(a,b,c,r,u,n)

      integer n
      integer nmax

      REAL_T a(n)
      REAL_T b(n)
      REAL_T c(n)
      REAL_T r(n)
      REAL_T u(n)

      parameter (nmax = 4098)

      integer j
      REAL_T bet
      REAL_T gam(nmax)
      if (b(1) .eq. 0) print *,'CANT HAVE B(1) = ZERO'

      bet = b(1)
      u(1) = r(1)/bet

      do j = 2,n
        gam(j) = c(j-1)/bet
        bet = b(j) - a(j)*gam(j)
        if (bet .eq. 0) then
          print *,'TRIDIAG FAILED '
          stop
        endif
        u(j) = (r(j)-a(j)*u(j-1))/bet
      enddo

      do j = n-1,1,-1
        u(j) = u(j) - gam(j+1)*u(j+1)
      enddo

      return
      end

c *************************************************************************
c ** MACPERIODIC **
c  Impose periodic boundary conditions on the single grid data in the
c   conjugate gradient bottom solver.
c *************************************************************************

      subroutine macperiodic(dest,DIMS,bc,ng)

      implicit none

      integer DIMS
      integer ng
      REAL_T   dest(lo_1-ng:hi_1+ng,lo_2-ng:hi_2+ng)
      integer bc(2,2)

      integer i,j

      if (BCX_LO .eq. PERIODIC .and. BCX_HI .eq. PERIODIC) then
        do j = lo_2,hi_2
          dest(lo_1-1,j) = dest(hi_1,j)
          dest(hi_1+1,j) = dest(lo_1,j)
        enddo
      endif
      if (BCY_LO .eq. PERIODIC .and. BCY_HI .eq. PERIODIC) then
        do i = lo_1,hi_1
          dest(i,lo_2-1) = dest(i,hi_2)
          dest(i,hi_2+1) = dest(i,lo_2)
        enddo
      endif

      return
      end
