#pragma once

#include <nall/suffix-array.hpp>
#include <nall/encode/bwt.hpp>
#include <nall/encode/huffman.hpp>
#include <nall/encode/mtf.hpp>
#include <nall/encode/rle.hpp>

namespace nall::Encode {

inline auto LZSA(std::span<const u8> input) -> std::vector<u8> {
  std::vector<u8> output;
  for(u32 byte : range(8)) output.push_back((u8)(input.size() >> (byte * 8)));

  auto suffixArray = SuffixArray(input).lpf();
  u32 index = 0;
  std::vector<u8> flags;
  std::vector<u8> literals;
  std::vector<u8> stringLengths;
  std::vector<u8> stringOffsets;

  u32 byte = 0, bits = 0;
  auto flagWrite = [&](bool bit) {
    byte = byte << 1 | bit;
    if(++bits == 8) flags.push_back((u8)byte), bits = 0;
  };

  auto literalWrite = [&](u8 literal) {
    literals.push_back(literal);
  };

  auto lengthWrite = [&](u64 length) {
         if(length < 1 <<  7) length = length << 1 |     0b1;
    else if(length < 1 << 14) length = length << 2 |    0b10;
    else if(length < 1 << 21) length = length << 3 |   0b100;
    else if(length < 1 << 28) length = length << 4 |  0b1000;
    else  /*length < 1 << 35*/length = length << 5 | 0b10000;
    while(length) stringLengths.push_back((u8)length), length >>= 8;
  };

  auto offsetWrite = [&](u32 offset) {
    stringOffsets.push_back((u8)(offset >>  0)); if(index < 1 <<  8) return;
    stringOffsets.push_back((u8)(offset >>  8)); if(index < 1 << 16) return;
    stringOffsets.push_back((u8)(offset >> 16)); if(index < 1 << 24) return;
    stringOffsets.push_back((u8)(offset >> 24));
  };

  while(index < input.size()) {
    s32 length, offset;
    suffixArray.previous(length, offset, index);

/*  for(u32 ahead = 1; ahead <= 2; ahead++) {
      s32 aheadLength, aheadOffset;
      suffixArray.previous(aheadLength, aheadOffset, index + ahead);
      if(aheadLength > length && aheadOffset >= 0) {
        length = 0;
        break;
      }
    } */

    if(length < 6 || offset < 0) {
      flagWrite(0);
      literalWrite(input[index++]);
    } else {
      flagWrite(1);
      lengthWrite(length - 6);
      offsetWrite(index - offset);
      index += length;
    }
  }
  while(bits) flagWrite(0);

  auto save = [&](const std::vector<u8>& buffer) {
    for(u32 byte : range(8)) output.push_back((u8)(buffer.size() >> (byte * 8)));
    std::ranges::copy(buffer, std::back_inserter(output));
  };

  save(Encode::Huffman({flags.data(), flags.size()}));
  save(Encode::Huffman({literals.data(), literals.size()}));
  save(Encode::Huffman({stringLengths.data(), stringLengths.size()}));
  save(Encode::Huffman({stringOffsets.data(), stringOffsets.size()}));

  return output;
}

}
