/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2004-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */


#ifndef INTERNAL_VERIFY_X509
# define INTERNAL_VERIFY_X509

#include <openssl/evp.h>
#include <openssl/x509.h>

#include "verify_x509_datatypes.h"


/************************************************************************
 * Type definitions
 ************************************************************************/

/**
 * Different functions using the error API. These function constants can be used
 * in ERR_PACK(), see man ERR_PACK
 */
typedef enum	{
    VER_F_VERIFY_X509_VERIFY	= 501,
    VER_F_PROCESS_INTERNAL	= 502,
    VER_F_GRID_VERIFYCERT	= 503,
    VER_F_SET_PARAM		= 504
} verify_func_t;


/************************************************************************
 * Macro definitions
 ************************************************************************/

/**
 * Calls \see verify_errval() with __FILE__ and __LINE__
 * \See verify_errval()
 */
#define VERIFY_errval(func, reason) \
    verify_errval(func, reason, __FILE__, __LINE__)

/**
 * Calls \see verify_reasonval() with __FILE__ and __LINE__
 * \See verify_reasonval()
 */
#define VERIFY_reasonval(func, reason) \
    verify_reasonval(func, reason, __FILE__, __LINE__)


/**
 * Checks whether input contains given type
 */
#define CERTISTYPE(input, type) \
    ( ( (input)&(type) ) == (type) )

/************************************************************************
 * Function prototypes
 ************************************************************************/

/**
 * Converts an internal library 'reason' for given function to an
 * ERR_get_error() compatible code and pushes the corresponding error on the
 * stack. Initializes the library when needed.
 * Returns the error code
 * */
unsigned long verify_errval(int func, verify_x509_error_t reason,
			    const char *file,int line);

/**
 * Converts an internal library 'reason' for given function to an
 * ERR_get_error() compatible code and pushes the corresponding error on the
 * stack. Initializes the library when needed.
 * Returns the reason code
 * */
verify_x509_error_t verify_reasonval(int func, verify_x509_error_t reason,
				     const char *file, int line);

/**
 * Initializes the library, by creating a new library ID and loading the error
 * and function strings.
 * Returns library number
 */
int verify_init_library(void);

/**
 * Reads the private key from filename and stores the result in pkey
 * Returns value of ERR_peek_error() on error or 0 on success.
 */
unsigned long verify_x509_readPrivateKeyFromFile(char *filename, EVP_PKEY **pkey);

/**
 * Reads the private key from PEM string and stores the result in pkey
 * Returns value of ERR_peek_error() on error or 0 on success.
 */
unsigned long verify_x509_readPrivateKeyFromPEM(char *pem, EVP_PKEY **pkey);

/**
 * Reads the certificate chain from filename and stores the result in certstack
 * Returns value of ERR_peek_error() on error or 0 on success.
 */
unsigned long verify_x509_readPublicCertChain(char *filename, STACK_OF(X509) **certstack);

/**
 * Verifies a privatekey with the first cert in the certstack
 * returns 0 on success, or appropriate value from ERR_peek_error().
 */
unsigned long verify_verifyPrivateKey(STACK_OF(X509) *certstack,EVP_PKEY *pkey);

/**
 * Validates a certificate with the CRL list and the signing CA
 * Return 0 on success or result of ERR_peek_error() on error (to be fixed)
 */
unsigned long verify_verifyCert( char * CA_DIR, STACK_OF(X509) *certstack, unsigned int verifyatnotbefore );

/**
 * converts proxy_type_t to human readable string
 */
const char *verify_certificate_type_str(proxy_type_t cert_type);

#endif /* INTERNAL_VERIFY_X509 */

